/*   Copyright Anritsu 2003  All Rights Reserved.                            */
/*****************************************************************************/

#include <formatio.h>
#include <visa.h>
#include <vpptype.h>
#include <ansi_c.h>
#include <string.h>
#include <analysis.h>
#include "MT8820.h"
//#include "MT8820 WCDMA LabWindows Driver.h"

#define AUMT8820_WCDMA_REVISION     "Rev 3.0,01Sep2003, CVI 5.5"     /* Instrument driver revision */

/* = MT8820 Radio Communication Analyzer (VISA) ======= */
/* 
    LabWindows 5.5 Instrument Driver
    Original Release: March 2003
    Modification History:

        Mar 2003 -  Original version
        Apr 2003 -  Added SEQOUTSW[?] command
        Jun 2003 -  SDM Changed lf to f and some char * to Char []
        Jul 2003 -  SDM Changed many errors (see word document)
     01 Sep 2003 -  [V3.0], Anritsu MCA1, New Released.  

*/
/* ========================================================================= */

/*****************************************************************************/
/*
       Function:   Set Active Marker
       Command(s): ACTMKR[?]
       Purpose:    Sets the marker to be activated in the time domain measured result window.
       Marker defined by:  AUMT8820_ACTIVE_MARKER_MARKER_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Active_Marker(ViSession instrSession, ViInt32 marker)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"MKR1", "MKR2", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   if (AuMT8820_InvalidViInt32Range(marker,0,1) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   len = sprintf(buf, "ACTMKR %s", commandArray[marker]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Active Marker
       Command(s): ACTMKR[?]
       Purpose:    Sets the marker to be activated in the time domain measured result window.
       Marker defined by:  AUMT8820_ACTIVE_MARKER_MARKER_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Active_Marker(ViSession instrSession, ViInt32  _VI_FAR *marker)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"MKR1", "MKR2", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViChar    resBuf0[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "ACTMKR?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   /* Get if numeric returned from instrument */
   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0) 
      return auMT8820_status;															   
   /* Scan(rdBuf, "%d", marker);  */

   /* Get if string returned from instrument */
   Scan(rdBuf, "%s", resBuf0); 
   if((auMT8820_status = AuMT8820_StringToEnum(commandArray, resBuf0, marker)) < 0) 
     return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Average Count for Adjacent Channel Power
       Command(s): ADJ_AVG[?]
       Purpose:    Set/Get the average count for the adjacent channel leakage power measurements
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Average_Count_for_Adjacent_Channel_Power(ViSession instrSession, ViInt32 count)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   if (AuMT8820_InvalidViInt32Range(count,1,9999) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   len = sprintf(buf, "ADJ_AVG %d", count);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Average Count for Adjacent Channel Power
       Command(s): ADJ_AVG[?]
       Purpose:    Set/Get the average count for the adjacent channel leakage power measurements
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Average_Count_for_Adjacent_Channel_Power(ViSession instrSession,  ViInt32 _VI_FAR *count)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "ADJ_AVG?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%d", count);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Adjacent Channel Power Mode
       Command(s): ADJ_MEAS[?]
       Purpose:    Set/Get the adjacent channel leakage power measurement mode
       Mode defined by:  AUMT8820_ADJACENT_CHANNEL_POWER_MODE_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Adjacent_Channel_Power_Mode(ViSession instrSession, ViInt32 mode)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"ON", "OFF", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   if (AuMT8820_InvalidViInt32Range(mode,0,1) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   len = sprintf(buf, "ADJ_MEAS %s", commandArray[mode]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Adjacent Channel Power Mode
       Command(s): ADJ_MEAS[?]
       Purpose:    Set/Get the adjacent channel leakage power measurement mode
       Mode defined by:  AUMT8820_ADJACENT_CHANNEL_POWER_MODE_xxx
 */
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Adjacent_Channel_Power_Mode(ViSession instrSession, ViInt32  _VI_FAR *mode)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"ON", "OFF", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViChar    resBuf0[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "ADJ_MEAS?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   /* Get if numeric returned from instrument */
   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0) 
      return auMT8820_status;															   
   /* Scan(rdBuf, "%d", mode);  */

   /* Get if string returned from instrument */
   Scan(rdBuf, "%s", resBuf0); 
   if((auMT8820_status = AuMT8820_StringToEnum(commandArray, resBuf0, mode)) < 0) 
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Downlink AICH Channelization Code
       Command(s): AICHCODE[?]
       Purpose:    Sets the channelization code of downlink AICH.
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Downlink_AICH_Channelization_Code(ViSession instrSession, ViInt32 code)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   if (AuMT8820_InvalidViInt32Range(code,0,255) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   len = sprintf(buf, "AICHCODE %d", code);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Downlink AICH Channelization Code
       Command(s): AICHCODE[?]
       Purpose:    Sets the channelization code of downlink AICH.
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Downlink_AICH_Channelization_Code(ViSession instrSession,  ViInt32 _VI_FAR *code)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "AICHCODE?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%d", code);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Downlink AICH Power
       Command(s): AICHPWR[?]
       Purpose:    Sets the output level of downlink AICH.
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Downlink_AICH_Power(ViSession instrSession, ViReal64 range)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   len = sprintf(buf, "AICHPWR %.1f", range);
   if (AuMT8820_InvalidViReal64Range(range, -30.000000, 0.000000) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Downlink AICH Power
       Command(s): AICHPWR[?]
       Purpose:    Sets the output level of downlink AICH.
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Downlink_AICH_Power(ViSession instrSession, ViReal64 _VI_FAR *range)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "AICHPWR?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%f", range);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Downlink AICH
       Command(s): AICH[?]
       Purpose:    Sets AICH output of the downlink physical channel to On or Off.
       Mode defined by:  AUMT8820_DOWNLINK_AICH_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Downlink_AICH(ViSession instrSession, ViInt32 mode)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"ON", "OFF", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   if (AuMT8820_InvalidViInt32Range(mode,0,1) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   len = sprintf(buf, "AICH %s", commandArray[mode]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Downlink AICH
       Command(s): AICH[?]
       Purpose:    Sets AICH output of the downlink physical channel to On or Off.
       Mode defined by:  AUMT8820_DOWNLINK_AICH_xxx
 */
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Downlink_AICH(ViSession instrSession, ViInt32  _VI_FAR *mode)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"ON", "OFF", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViChar    resBuf0[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "AICH?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   /* Get if numeric returned from instrument */
   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0) 
      return auMT8820_status;															   
   /* Scan(rdBuf, "%d", mode);  */

   /* Get if string returned from instrument */
   Scan(rdBuf, "%s", resBuf0); 
   if((auMT8820_status = AuMT8820_StringToEnum(commandArray, resBuf0, mode)) < 0) 
     return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set AF Input Full Scale
       Command(s): AIFLSCL[?]
       Purpose:    Sets the full scale value of the audio input.
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_AF_Input_Full_Scale(ViSession instrSession, ViReal64 range_mV)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   len = sprintf(buf, "AIFLSCL %.2f", range_mV);
   if (AuMT8820_InvalidViReal64Range(range_mV, 1.000000, 5000.000000) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get AF Input Full Scale
       Command(s): AIFLSCL[?]
       Purpose:    Sets the full scale value of the audio input.
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_AF_Input_Full_Scale(ViSession instrSession, ViReal64 _VI_FAR *range_mV)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "AIFLSCL?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%f", range_mV);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Encoder Input Gain
       Command(s): AIGAIN[?]
       Purpose:    Sets the audio input gain

*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Encoder_Input_Gain_WCDMA(ViSession instrSession, ViReal64 gain_dB)
{
  ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   len = sprintf(buf, "AIGAIN %.2f", gain_dB);
   if (AuMT8820_InvalidViReal64Range(gain_dB, -3.000000, 3.000000) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Encoder Input Gain
       Command(s): AIGAIN[?]
       Purpose:    Sets the audio input gain

*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Encoder_Input_Gain_WCDMA(ViSession instrSession, ViReal64 _VI_FAR *gain_dB)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "AIGAIN?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%f", gain_dB);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Audio Input Output Connector
       Command(s): AINOUT[?]
       Purpose:    Sets/Gets the audio I/O terminal to the handset connector.
       Connector defined by:  AUMT8820_AUDIO_INPUT_OUTPUT_CONNECTOR_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Audio_Input_Output_Connector_WCDMA(ViSession instrSession, ViInt32 connector)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"AF", "HANDSET", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   if (AuMT8820_InvalidViInt32Range(connector,0,1) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   len = sprintf(buf, "AINOUT %s", commandArray[connector]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Audio Input Output Connector
       Command(s): AINOUT[?]
       Purpose:    Sets/Gets the audio I/O terminal to the handset connector.
       Connector defined by:  AUMT8820_AUDIO_INPUT_OUTPUT_CONNECTOR_xxx
 */
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Audio_Input_Output_Connector_WCDMA(ViSession instrSession, ViInt32  _VI_FAR *connector)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"AF", "HANDSET", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViChar    resBuf0[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "AINOUT?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   /* Get if numeric returned from instrument */
   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0) 
       return auMT8820_status;															   
   /* Scan(rdBuf, "%d", connector);	 */

   /* Get if string returned from instrument */
   Scan(rdBuf, "%s", resBuf0); 
   if((auMT8820_status = AuMT8820_StringToEnum(commandArray, resBuf0, connector)) < 0) 
     return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get All Fundamental Measurement Results WCDMA
       Command(s): ALLMEAS?
       Purpose:    Get all fundamental measurement results
       Mode defined by:  AUMT8820_ALL_FUNDAMENTAL_MEASUREMENT_RESULTS_WCDMA_xxx
 */
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_All_Fundamental_Measurement_Results_WCDMA(ViSession instrSession, ViChar _VI_FAR results[], ViInt32 _VI_FAR tag)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString comandArray[] = {"", "PWR", "FREQ", "OBW", "SMASK", "ADJ", "MOD", "PCDE", "BER", "BLER", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[LARGE_BUFFER_SIZE];
   ViChar    resBuf0[LARGE_BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "ALLMEAS? %s",comandArray[tag]);  
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   /* Get if numeric returned from instrument */
   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, LARGE_BUFFER_M1, &bytesRead)) < 0) 
      return auMT8820_status;															   
   Scan(rdBuf, "%s", results);  															   

   /* Get if string returned from instrument */
   /*  Scan(rdBuf, "%s", results); */
   /*  if((auMT8820_status = AuMT8820_StringToEnum(commandArray, resBuf0, results)) < 0) */
   /*    return auMT8820_status;*/

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set All Fundamental Measurement Items WCDMA
       Command(s): ALLMEASITEMS[?]
       Purpose:    Set/Get the fundamental measurement items
       Mode defined by:  AUMT8820_ALL_FUNDAMENTAL_MEASUREMENT_ITEMS_WCDMA_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_All_Fundamental_Measurement_Items_WCDMA(ViSession instrSession, ViChar *mode)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   // static ViString commandArray[] = {"ON", "OFF", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   len = sprintf(buf, "ALLMEASITEMS %s", mode);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get All Fundamental Measurement Items WCDMA
       Command(s): ALLMEASITEMS[?]
       Purpose:    Set/Get the fundamental measurement items
       Mode defined by:  AUMT8820_ALL_FUNDAMENTAL_MEASUREMENT_ITEMS_WCDMA_xxx
 */
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_All_Fundamental_Measurement_Items_WCDMA(ViSession instrSession, ViChar  _VI_FAR mode[])
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   // static ViString commandArray[] = {"ON", "OFF", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViChar    resBuf0[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "ALLMEASITEMS?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   /* Get if numeric returned from instrument */
   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%s", mode);

   /* Get if string returned from instrument */
   /*  Scan(rdBuf, "%s", resBuf0); */
   /*  if((auMT8820_status = AuMT8820_StringToEnum(commandArray, resBuf0, mode)) < 0) */
   /*    return auMT8820_status;*/

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Handset Microphone Volume
       Command(s): AMICVOL[?]
       Purpose:    Sets/Gets the handset microphone volume.

*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Handset_Microphone_Volume_WCDMA(ViSession instrSession, ViInt32 volume)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   if (AuMT8820_InvalidViInt32Range(volume,0,5) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   len = sprintf(buf, "AMICVOL %d", volume);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Handset Microphone Volume
       Command(s): AMICVOL[?]
       Purpose:    Sets/Gets the handset microphone volume.

*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Handset_Microphone_Volume_WCDMA(ViSession instrSession,  ViInt32 _VI_FAR *volume)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "AMICVOL?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%d", volume);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set AF Output Full Scale
       Command(s): AOFLSCL[?]
       Purpose:    Sets the full scale value of the audio output.
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_AF_Output_Full_Scale(ViSession instrSession, ViReal64 range_mV)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   if (AuMT8820_InvalidViReal64Range(range_mV, 0.000000, 5000.000000) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   if (range_mV >= 0.00 && range_mV <= 50.00)
      len = sprintf(buf, "AOFLSCL %.2f", range_mV);
   if (range_mV > 50.00 && range_mV <= 500.00)
      len = sprintf(buf, "AOFLSCL %.1f", range_mV);
   if (range_mV > 500.00 && range_mV <= 5000.00)
      len = sprintf(buf, "AOFLSCL %.0f", range_mV);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get AF Output Full Scale
       Command(s): AOFLSCL[?]
       Purpose:    Sets the full scale value of the audio output.
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_AF_Output_Full_Scale(ViSession instrSession, ViReal64 _VI_FAR *range_mV)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "AOFLSCL?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%f", range_mV);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Handset Speaker Volume
       Command(s): ASPVOL[?]
       Purpose:    Sets/Gets the handset speaker volume.
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Handset_Speaker_Volume_WCDMA(ViSession instrSession, ViInt32 volume)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   if (AuMT8820_InvalidViInt32Range(volume,0,5) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   len = sprintf(buf, "ASPVOL %d", volume);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Handset Speaker Volume
       Command(s): ASPVOL[?]
       Purpose:    Sets/Gets the handset speaker volume.
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Handset_Speaker_Volume_WCDMA(ViSession instrSession,  ViInt32 _VI_FAR *volume)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "ASPVOL?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%d", volume);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Authentication Key
       Command(s): AUTHENT_KEY[?]
       Purpose:    Sets 32 bits of the authentication key for a phone.
       Part defined by:  AUMT8820_AUTHENTICATION_KEY_VALUE_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Authentication_Key(ViSession instrSession, ViInt32 part, ViChar *value)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"1", "2", "3", "4", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
//   if (AuMT8820_InvalidViInt32Range(value,0x0,0x0FFFFFFFF) == VI_TRUE)
//      return VI_ERROR_OUTOF_RANGE;
   if (AuMT8820_InvalidViInt32Range(part,0,3) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;

   len = sprintf(buf, "AUTHENT_KEY %s,%s", commandArray[part],value);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Authentication Key
       Command(s): AUTHENT_KEY[?]
       Purpose:    Sets 32 bits of the authentication key for a phone.
       Part defined by:  AUMT8820_AUTHENTICATION_KEY_VALUE_xxx
 */
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Authentication_Key(ViSession instrSession, ViInt32 part, ViChar  *value)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"1", "2", "3", "4", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViChar    resBuf0[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "AUTHENT_KEY? %s", commandArray[part]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   /* Get if numeric returned from instrument */
   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%s", value);

   /* Get if string returned from instrument */
   /*  Scan(rdBuf, "%s", resBuf0); */
   /*  if((auMT8820_status = AuMT8820_StringToEnum(commandArray, resBuf0, range)) < 0) */
   /*    return auMT8820_status;*/

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Authentication Key All
       Command(s): AUTHENT_KEYALL[?]
       Purpose:    Sets all bits of the authentication key for a phone.
       Mode defined by:  AUMT8820_AUTHENTICATION_KEY_ALL_VALUE_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Authentication_Key_All(ViSession instrSession, ViChar *value1, ViChar *value2, ViChar *value3, ViChar *value4)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   // static ViString commandArray[] = {"1", "2", "3", "4", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
/*   
   if (AuMT8820_InvalidViInt32Range(value1,0x00000000,0xFFFFFFFF) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   if (AuMT8820_InvalidViInt32Range(value2,0x00000000,0xFFFFFFFF) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   if (AuMT8820_InvalidViInt32Range(value3,0x00000000,0xFFFFFFFF) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   if (AuMT8820_InvalidViInt32Range(value4,0x00000000,0xFFFFFFFF) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
*/
   len = sprintf(buf, "AUTHENT_KEYALL %s,%s,%s,%s", value1, value2, value3, value4);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Authentication Key All
       Command(s): AUTHENT_KEYALL[?]
       Purpose:    Sets all bits of the authentication key for a phone.
       Mode defined by:  AUMT8820_AUTHENTICATION_KEY_ALL_VALUE_xxx
 */
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Authentication_Key_All(ViSession instrSession, ViChar *value)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   // static ViString commandArray[] = {"1", "2", "3", "4", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViChar    resBuf0[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "AUTHENT_KEYALL?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   /* Get if numeric returned from instrument */
   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%s", value);

   /* Get if string returned from instrument */
   /*  Scan(rdBuf, "%s", resBuf0); */
   /*  if((auMT8820_status = AuMT8820_StringToEnum(commandArray, resBuf0, range)) < 0) */
   /*    return auMT8820_status;*/

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set AUX External Loss
       Command(s): AUEXTLOSS[?]
       Purpose:    Sets the external loss at the AUX output.
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_AUX_External_Loss(ViSession instrSession, ViReal64 level_dB)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   len = sprintf(buf, "AUEXTLOSS %.1f", level_dB);
   if (AuMT8820_InvalidViReal64Range(level_dB, 0.000000, 55.000000) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get AUX External Loss
       Command(s): AUEXTLOSS[?]
       Purpose:    Sets the external loss at the AUX output.
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_AUX_External_Loss(ViSession instrSession, ViReal64 _VI_FAR *level_dB)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "AUEXTLOSS?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%f", level_dB);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set AUX External Loss On Off
       Command(s): AUEXTLOSSW[?]
       Purpose:    Sets the AUX output external loss setting to On or Off.
       Mode defined by:  AUMT8820_AUX_EXTERNAL_LOSS_MODE_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_AUX_External_Loss_On_Off(ViSession instrSession, ViInt32 mode)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"ON", "OFF", "COMMON", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   if (AuMT8820_InvalidViInt32Range(mode, 0, 2) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   len = sprintf(buf, "AUEXTLOSSW %s", commandArray[mode]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get AUX External Loss On Off
       Command(s): AUEXTLOSSW[?]
       Purpose:    Sets the AUX output external loss setting to On or Off.
       Mode defined by:  AUMT8820_AUX_EXTERNAL_LOSS_MODE_xxx
 */
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_AUX_External_Loss_On_Off(ViSession instrSession, ViInt32  _VI_FAR *mode)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"ON", "OFF", "COMMON", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViChar    resBuf0[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "AUEXTLOSSW?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   /* Get if numeric returned from instrument */
   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   /* Scan(rdBuf, "%d", mode); */

   /* Get if string returned from instrument */
   Scan(rdBuf, "%s", resBuf0); 
   if((auMT8820_status = AuMT8820_StringToEnum(commandArray, resBuf0, mode)) < 0) 
     return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Carrier Frequency Average
       Command(s): AVG_CARRF?
       Purpose:    Gets the measured result for the carrier frequency average using average count
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Carrier_Frequency_Average_WCDMA(ViSession instrSession, ViReal64 _VI_FAR *frequency_Hz)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "AVG_CARRF?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%f", frequency_Hz);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Carrier Frequency Error Average
       Command(s): AVG_CARRFERR?
       Purpose:    Get the average carrier frequency error
       Unit defined by:  AUMT8820_CARRIER_FREQ_ERROR_UNIT_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Carrier_Frequency_Error_Average_WCDMA(ViSession instrSession, ViInt32 unit, ViReal64 _VI_FAR *error)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"HZ", "PPM", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   if (AuMT8820_InvalidViInt32Range(unit, 0, 1) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   len = sprintf(buf, "AVG_CARRFERR? %s", commandArray[unit]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%f", error);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Error Vector Magnitude Average
       Command(s): AVG_EVM?
       Purpose:    Get the average error vector magnitude in percent
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Error_Vector_Magnitude_Average(ViSession instrSession, ViReal64 _VI_FAR *percent)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "AVG_EVM?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%f", percent);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Filtered Power Average
       Command(s): AVG_FILTPWR?
       Purpose:    Queries the average value of the measured result for RRC filtered transmission power.
       Unit defined by:  AUMT8820_FILTERED_POWER_UNIT_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Filtered_Power_Average(ViSession instrSession, ViInt32 unit, ViReal64 _VI_FAR *power)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"DBM", "WATT", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   if (AuMT8820_InvalidViInt32Range(unit, 0, 1) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   len = sprintf(buf, "AVG_FILTPWR? %s", commandArray[unit]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%f", power);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get IQ Imbalance Average
       Command(s): AVG_IQIMB?
       Purpose:    Get the average IQ imbalance
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_IQ_Imbalance_Average(ViSession instrSession, ViReal64 _VI_FAR *imbalance)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "AVG_IQIMB?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%f", imbalance);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Magnitude Error Average
       Command(s): AVG_MAGTDERR?
       Purpose:    Queries the value of magnitude error measurement results of averaging.
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Magnitude_Error_Average_WCDMA(ViSession instrSession, ViReal64 _VI_FAR *percent)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "AVG_MAGTDERR?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%f", percent);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Adjacent Channel Power Average
       Command(s): AVG_MODPWR?
       Purpose:    Get the average adjacent channel leakage power level
       Offset defined by:  AUMT8820_ADJACENT_CHAN_POWER_OFFSET_xxx
       Unit defined by:  AUMT8820_ADJACENT_CHAN_POWER_UNIT_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Adjacent_Channel_Power_Average(ViSession instrSession, ViInt32 offset, ViInt32 unit, ViReal64 _VI_FAR *level)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"LOW10", "LOW5", "UP5", "UP10", 0L};
   static ViString commandArray2[] = {"DB", "DBM", "WATT", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   if (AuMT8820_InvalidViInt32Range(offset, 0, 3) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   if (AuMT8820_InvalidViInt32Range(unit, 0, 2) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   len = sprintf(buf, "AVG_MODPWR? %s,%s",commandArray[offset], commandArray2[unit]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%f", level);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Origin Offset Average
       Command(s): AVG_ORGNOFS?
       Purpose:    Get the average origin offset in dB
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Origin_Offset_Average(ViSession instrSession, ViReal64 _VI_FAR *offset_dB)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "AVG_ORGNOFS?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%f", offset_dB);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Peak Code Domain Error Average
       Command(s): AVG_PCDERR?
       Purpose:    Queries the average value of the measured result for peak code domain measurement.
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Peak_Code_Domain_Error_Average(ViSession instrSession, ViReal64 _VI_FAR *level_dB)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "AVG_PCDERR?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%f", level_dB);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get RMS Phase Vector Error Average
       Command(s): AVG_PHASEERR?
       Purpose:    Gets the average value among measured results for average count of phase vector error.
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_RMS_Phase_Vector_Error_Average(ViSession instrSession, ViReal64 _VI_FAR *degrees)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "AVG_PHASEERR?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%f", degrees);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get TX Power Average WCDMA
       Command(s): AVG_POWER?
       Purpose:    Get the average value among the measured results for average count of transmission power measurements for a phone
       Unit defined by:  AUMT8820_TX_POWER_UNIT_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_TX_Power_Average_WCDMA(ViSession instrSession, ViInt32 unit, ViReal64 _VI_FAR *power)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"DBM", "WATT", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   if (AuMT8820_InvalidViInt32Range(unit, 0, 1) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   len = sprintf(buf, "AVG_POWER? %s", commandArray[unit]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%f", power);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Peak Vector Error Average
       Command(s): AVG_PVECTERR?
       Purpose:    Queries the average value of the measured result for peak vector error measurement.
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Peak_Vector_Error_Average(ViSession instrSession, ViReal64 _VI_FAR *percent)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "AVG_PVECTERR?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%f", percent);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Timing Error Average
       Command(s): AVG_TMGERR?
       Purpose:    Queries the average value of the measured result for timing error measurement.
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Timing_Error_Average(ViSession instrSession, ViReal64 _VI_FAR *chip)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "AVG_TMGERR?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%f", chip);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set AWGN Output Mode
       Command(s): AWGNLVL[?]
       Purpose:    Sets the AWGN setting to On or Off.
       Setting defined by:  AUMT8820_AWGN_OUTPUT_MODE_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_AWGN_Output_Mode(ViSession instrSession, ViInt32 setting)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"ON", "OFF", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   if (AuMT8820_InvalidViInt32Range(setting, 0, 1) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   len = sprintf(buf, "AWGNLVL %s", commandArray[setting]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get AWGN Output Mode
       Command(s): AWGNLVL[?]
       Purpose:    Sets the AWGN setting to On or Off.
       Setting defined by:  AUMT8820_AWGN_OUTPUT_MODE_xxx
 */
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_AWGN_Output_Mode(ViSession instrSession, ViInt32  _VI_FAR *setting)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"ON", "OFF", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViChar    resBuf0[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "AWGNLVL?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   /* Get if numeric returned from instrument */
   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   /* Scan(rdBuf, "%d", setting); */

   /* Get if string returned from instrument */
   Scan(rdBuf, "%s", resBuf0); 
   if((auMT8820_status = AuMT8820_StringToEnum(commandArray, resBuf0, setting)) < 0) 
     return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set AWGN Output
       Command(s): AWGNPWR[?]
       Purpose:    Sets the output level for AWGN
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_AWGN_Output(ViSession instrSession, ViReal64 range_dB)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   len = sprintf(buf, "AWGNPWR %.1fDB", range_dB);
   if (AuMT8820_InvalidViReal64Range(range_dB, -20.000000, 5.000000) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get AWGN Output
       Command(s): AWGNPWR[?]
       Purpose:    Sets the output level for AWGN
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_AWGN_Output(ViSession instrSession, ViReal64 _VI_FAR *range_dB)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "AWGNPWR?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%f", range_dB);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Band Calibration
       Command(s): BANDCAL
       Purpose:    Performs calibration within the W-CDMA frequency bandwidth
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Band_Calibration_WCDMA(ViSession instrSession)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   len = sprintf(buf, "BANDCAL");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Base Slot Setup
       Command(s): BASESLOT[?]
       Purpose:    Sets the base slot as a reference of relative value display for a slot list window in the time domain measurement area.
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Base_Slot_Setup(ViSession instrSession, ViInt32 slot)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   if (AuMT8820_InvalidViInt32Range(slot,-15,164) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   len = sprintf(buf, "BASESLOT %d", slot);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Base Slot Setup
       Command(s): BASESLOT[?]
       Purpose:    Sets the base slot as a reference of relative value display for a slot list window in the time domain measurement area.
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Base_Slot_Setup(ViSession instrSession,  ViInt32 _VI_FAR *slot)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "BASESLOT?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%d", slot);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get BCCH Signaling
       Command(s): BCCH?
       Purpose:    Queries BCCH transmission rate for signaling RB's.
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_BCCH_Signaling(ViSession instrSession, ViReal64 _VI_FAR *rate)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "BCCH?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%f", rate);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Bit Error Rate Count
       Command(s): BERCNT?
       Purpose:    Get the number of error bits 
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Bit_Error_Rate_Count_WCDMA(ViSession instrSession, ViInt32 _VI_FAR *count)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "BERCNT?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%d", count);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Bit Error Rate Judgement
       Command(s): BERPASS?
       Purpose:    Get the judgement (Pass/Fail) of the result for bit error rate measurement
       Unit defined by:  AUMT8820_BIT_ERROR_RATE_JUDGEMENT_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Bit_Error_Rate_Judgement(ViSession instrSession, ViInt32 _VI_FAR *judgement)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"PASS", "FAIL", "-", "+", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViChar    resBuf0[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "BERPASS?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   /* Get if numeric returned from instrument */
   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   /* Scan(rdBuf, "%d", judgement); */

   /* Get if string returned from instrument */
   Scan(rdBuf, "%s", resBuf0); 
   if((auMT8820_status = AuMT8820_StringToEnum(commandArray, resBuf0, judgement)) < 0) 
     return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Bit Error Rate Transmit
       Command(s): BERTRANSMIT?
       Purpose:    Queries the number of transmited bits during bit error rate measurement.
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Bit_Error_Rate_Transmit(ViSession instrSession, ViInt32 _VI_FAR *transmitted_Bits)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "BERTRANSMIT?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%d", transmitted_Bits);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Bit Error Rate Ratio WCDMA
       Command(s): BER?
       Purpose:    Get the  bit error rate ratio 
       Unit defined by:  AUMT8820_BIT_ERROR_RATE_WCDMA_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Bit_Error_Rate_Ratio_WCDMA(ViSession instrSession,ViInt32 unit, ViReal64 _VI_FAR *result)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"", "PER", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   if (AuMT8820_InvalidViInt32Range(unit, 0, 1) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   len = sprintf(buf, "BER? %s", commandArray[unit]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%f", result);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Bit Error Rate Input Data Clock Edge
       Command(s): BER_INPUTCLK[?]
       Purpose:    Sets/Gets the start point of the external input data for a bit error rate measurement.
       Clock_Edge defined by:  AUMT8820_BIT_ERROR_RATE_INPUT_DATA_CLOCK_EDGE_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Bit_Error_Rate_Input_Data_Clock_Edge_WCDMA(ViSession instrSession, ViInt32 clock_Edge)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"RISE", "FALL", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   if (AuMT8820_InvalidViInt32Range(clock_Edge, 0, 1) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   len = sprintf(buf, "BER_INPUTCLK %s", commandArray[clock_Edge]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Bit Error Rate Input Data Clock Edge
       Command(s): BER_INPUTCLK[?]
       Purpose:    Sets/Gets the start point of the external input data for a bit error rate measurement.
       Clock_Edge defined by:  AUMT8820_BIT_ERROR_RATE_INPUT_DATA_CLOCK_EDGE_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Bit_Error_Rate_Input_Data_Clock_Edge_WCDMA(ViSession instrSession, ViInt32  _VI_FAR *clock_Edge)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"RISE", "FALL", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViChar    resBuf0[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "BER_INPUTCLK?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   /* Get if numeric returned from instrument */
   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   /* Scan(rdBuf, "%d", clock_Edge); */

   /* Get if string returned from instrument */
   Scan(rdBuf, "%s", resBuf0); 
   if((auMT8820_status = AuMT8820_StringToEnum(commandArray, resBuf0, clock_Edge)) < 0) 
     return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Bit Error Rate Input Polarity
       Command(s): BER_INPUTPOL[?]
       Purpose:    Sets/Gets the  data polarity for a bit error rate measurement.
       Polarity defined by:  AUMT8820_BIT_ERROR_RATE_INPUT_POLARITY_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Bit_Error_Rate_Input_Polarity_WCDMA(ViSession instrSession, ViInt32 polarity)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"POS", "NEG", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   if (AuMT8820_InvalidViInt32Range(polarity, 0, 1) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   len = sprintf(buf, "BER_INPUTPOL %s", commandArray[polarity]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Bit Error Rate Input Polarity
       Command(s): BER_INPUTPOL[?]
       Purpose:    Sets/Gets the  data polarity for a bit error rate measurement.
       Polarity defined by:  AUMT8820_BIT_ERROR_RATE_INPUT_POLARITY_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Bit_Error_Rate_Input_Polarity_WCDMA(ViSession instrSession, ViInt32  _VI_FAR *polarity)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"POS", "NEG", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViChar    resBuf0[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "BER_INPUTPOL?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   /* Get if numeric returned from instrument */
   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   /* Scan(rdBuf, "%d", polarity); */

   /* Get if string returned from instrument */
   Scan(rdBuf, "%s", resBuf0); 
   if((auMT8820_status = AuMT8820_StringToEnum(commandArray, resBuf0, polarity)) < 0) 
     return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Bit Error Rate Upper Limit
       Command(s): BER_LIMIT[?]
       Purpose:    Get/Set the maximum value for the bit error rate measurement pass/fail template
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Bit_Error_Rate_Upper_Limit(ViSession instrSession, ViReal64 ratio)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   len = sprintf(buf, "BER_LIMIT %.1f", ratio);
   if (AuMT8820_InvalidViReal64Range(ratio, 0.000000, 100.000000) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Bit Error Rate Upper Limit
       Command(s): BER_LIMIT[?]
       Purpose:    Get/Set the maximum value for the bit error rate measurement pass/fail template
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Bit_Error_Rate_Upper_Limit(ViSession instrSession, ViReal64 _VI_FAR *ratio)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "BER_LIMIT?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%f", ratio);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Bit Error Rate Measurement Mode
       Command(s): BER_MEAS[?]
       Purpose:    Set/Get the bit error rate (BER) measurement mode
       Mode defined by:  AUMT8820_BIT_ERROR_RATE_MEASUREMENT_MODE_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Bit_Error_Rate_Measurement_Mode_WCDMA(ViSession instrSession, ViInt32 mode)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"ON", "OFF", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   if (AuMT8820_InvalidViInt32Range(mode, 0, 1) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   len = sprintf(buf, "BER_MEAS %s", commandArray[mode]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Bit Error Rate Measurement Mode
       Command(s): BER_MEAS[?]
       Purpose:    Set/Get the bit error rate (BER) measurement mode
       Mode defined by:  AUMT8820_BIT_ERROR_RATE_MEASUREMENT_MODE_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Bit_Error_Rate_Measurement_Mode_WCDMA(ViSession instrSession, ViInt32  _VI_FAR *mode)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"ON", "OFF", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViChar    resBuf0[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "BER_MEAS?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   /* Get if numeric returned from instrument */
   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   /* Scan(rdBuf, "%d", mode); */

   /* Get if string returned from instrument */
   Scan(rdBuf, "%s", resBuf0); 
   if((auMT8820_status = AuMT8820_StringToEnum(commandArray, resBuf0, mode)) < 0) 
     return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Bit Error Rate Measurement Input
       Command(s): BER_MEASINPUT[?]
       Purpose:    Sets/Gets the measurement object for a bit error rate measurement.
       Input defined by:  AUMT8820_BIT_ERROR_RATE_MEASUREMENT_INPUT_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Bit_Error_Rate_Measurement_Input_WCDMA(ViSession instrSession, ViInt32 input)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"RF", "EXT", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   if (AuMT8820_InvalidViInt32Range(input, 0, 1) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   len = sprintf(buf, "BER_MEASINPUT %s", commandArray[input]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Bit Error Rate Measurement Input
       Command(s): BER_MEASINPUT[?]
       Purpose:    Sets/Gets the measurement object for a bit error rate measurement.
       Input defined by:  AUMT8820_BIT_ERROR_RATE_MEASUREMENT_INPUT_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Bit_Error_Rate_Measurement_Input_WCDMA(ViSession instrSession, ViInt32  _VI_FAR *input)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"RF", "EXT", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViChar    resBuf0[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "BER_MEASINPUT?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   /* Get if numeric returned from instrument */
   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   /* Scan(rdBuf, "%d", input); */

   /* Get if string returned from instrument */
   Scan(rdBuf, "%s", resBuf0); 
   if((auMT8820_status = AuMT8820_StringToEnum(commandArray, resBuf0, input)) < 0) 
     return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Bit Error Rate Number Of Samples WCDMA
       Command(s): BER_SAMPLE[?]
       Purpose:    Get/Set the number of sample bits for the bit error rate measurement
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Bit_Error_Rate_Number_Of_Samples_WCDMA(ViSession instrSession, ViInt32 samples)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   if (AuMT8820_InvalidViInt32Range(samples,1,99999999) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   len = sprintf(buf, "BER_SAMPLE %d", samples);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Bit Error Rate Number Of Samples WCDMA
       Command(s): BER_SAMPLE[?]
       Purpose:    Get/Set the number of sample bits for the bit error rate measurement
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Bit_Error_Rate_Number_Of_Samples_WCDMA(ViSession instrSession,  ViInt32 _VI_FAR *samples)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "BER_SAMPLE?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%d", samples);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Block Error Rate WCDMA
       Command(s): BLER?
       Purpose:    Get the block error rate ratio
       Unit defined by:  AUMT8820_BLOCK_ERROR_RATE_WCDMA_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Block_Error_Rate_WCDMA(ViSession instrSession, ViInt32 unit, ViReal64 _VI_FAR *ratio)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString comandArray[] = {"", "PER", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   if (AuMT8820_InvalidViInt32Range(unit, 0, 1) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   len = sprintf(buf, "BLER? %s", comandArray[unit]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%f", ratio);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Block Error Rate Count
       Command(s): BLERCNT?
       Purpose:    Get the number of block error bits
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Block_Error_Rate_Count_WCDMA(ViSession instrSession, ViInt32 _VI_FAR *count)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "BLERCNT?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%d", count);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Block Error Rate Judgement
       Command(s): BLERPASS?
       Purpose:    Get the judgement (Pass/Fail) of the result for the block error rate measurement
       Judgement defined by:  AUMT8820_BLOCK_ERROR_RATE_JUDGEMENT_xxx
 */
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Block_Error_Rate_Judgement(ViSession instrSession, ViInt32 _VI_FAR *judgement)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"PASS", "FAIL", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViChar    resBuf0[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "BLERPASS?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   /* Get if numeric returned from instrument */
   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   /* Scan(rdBuf, "%d", judgement); */

   /* Get if string returned from instrument */
   Scan(rdBuf, "%s", resBuf0); 
   if((auMT8820_status = AuMT8820_StringToEnum(commandArray, resBuf0, judgement)) < 0) 
     return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Block Error Rate Transmit
       Command(s): BLERTRANSMIT?
       Purpose:    Queries the number of transmited blocks during bit error rate measurement.
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Block_Error_Rate_Transmit(ViSession instrSession, ViInt32 _VI_FAR *transmitted_Blocks)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "BLERTRANSMIT?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%d", transmitted_Blocks);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Block Error Rate Upper Limit
       Command(s): BLER_LIMIT[?]
       Purpose:    Get/Set the  maximum value for the block error rate measurement
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Block_Error_Rate_Upper_Limit(ViSession instrSession, ViReal64 limit)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   len = sprintf(buf, "BLER_LIMIT %.1f", limit);
   if (AuMT8820_InvalidViReal64Range(limit, 0.000000, 100.000000) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Block Error Rate Upper Limit
       Command(s): BLER_LIMIT[?]
       Purpose:    Get/Set the  maximum value for the block error rate measurement
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Block_Error_Rate_Upper_Limit(ViSession instrSession, ViReal64 _VI_FAR *limit)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "BLER_LIMIT?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%f", limit);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Block Error Rate Measurement Mode
       Command(s): BLER_MEAS[?]
       Purpose:    Set/Get the block error rate (BLER) measurement mode
       Mode defined by:  AUMT8820_BLOCK_ERROR_RATE_MEASUREMENT_MODE_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Block_Error_Rate_Measurement_Mode_WCDMA(ViSession instrSession, ViInt32 mode)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"ON", "OFF", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   if (AuMT8820_InvalidViInt32Range(mode, 0, 1) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   len = sprintf(buf, "BLER_MEAS %s", commandArray[mode]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Block Error Rate Measurement Mode
       Command(s): BLER_MEAS[?]
       Purpose:    Set/Get the block error rate (BLER) measurement mode
       Mode defined by:  AUMT8820_BLOCK_ERROR_RATE_MEASUREMENT_MODE_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Block_Error_Rate_Measurement_Mode_WCDMA(ViSession instrSession, ViInt32  _VI_FAR *mode)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"ON", "OFF", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViChar    resBuf0[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "BLER_MEAS?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   /* Get if numeric returned from instrument */
   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   /* Scan(rdBuf, "%d", mode);*/

   /* Get if string returned from instrument */
   Scan(rdBuf, "%s", resBuf0); 
   if((auMT8820_status = AuMT8820_StringToEnum(commandArray, resBuf0, mode)) < 0) 
     return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Block Error Rate Number Of Samples
       Command(s): BLER_SAMPLE[?]
       Purpose:    Get/Set the number of sample blocks for the block error rate measurement
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Block_Error_Rate_Number_Of_Samples_WCDMA(ViSession instrSession, ViInt32 samples)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   if (AuMT8820_InvalidViInt32Range(samples,1,99999999) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   len = sprintf(buf, "BLER_SAMPLE %d", samples);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Block Error Rate Number Of Samples
       Command(s): BLER_SAMPLE[?]
       Purpose:    Get/Set the number of sample blocks for the block error rate measurement
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Block_Error_Rate_Number_Of_Samples_WCDMA(ViSession instrSession,  ViInt32 _VI_FAR *samples)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "BLER_SAMPLE?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%d", samples);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Call Drop
       Command(s): CALLDROP[?]
       Purpose:    Sets the call drop setting to On or Off.
       Setting defined by:  AUMT8820_CALL_DROP_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Call_Drop(ViSession instrSession, ViInt32 setting)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"ON", "OFF", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   if (AuMT8820_InvalidViInt32Range(setting, 0, 1) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   len = sprintf(buf, "CALLDROP %s", commandArray[setting]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Call Drop
       Command(s): CALLDROP[?]
       Purpose:    Sets the call drop setting to On or Off.
       Setting defined by:  AUMT8820_CALL_DROP_xxx
 */
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Call_Drop(ViSession instrSession, ViInt32  _VI_FAR *setting)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"ON", "OFF", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViChar    resBuf0[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "CALLDROP?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   /* Get if numeric returned from instrument */
   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   /* Scan(rdBuf, "%d", setting); */

   /* Get if string returned from instrument */
   Scan(rdBuf, "%s", resBuf0); 
   if((auMT8820_status = AuMT8820_StringToEnum(commandArray, resBuf0, setting)) < 0) 
     return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Call Processing Error WCDMA
       Command(s): CALLERR?
       Purpose:    Get the error sequence when a call processing failed
       Code defined by:  AUMT8820_CALL_PROCESSING_WCDMA_xxx
 */
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Call_Processing_Error_WCDMA(ViSession instrSession, ViInt32 _VI_FAR *code, ViInt32 _VI_FAR *error)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   //static ViString comandArray[] = {"", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViChar    resBuf0[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "CALLERR?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   /* Get if numeric returned from instrument */
   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%d,%d", code, error);

   /* Get if string returned from instrument */
   /*  Scan(rdBuf, "%s", resBuf0); */
   /*  if((auMT8820_status = AuMT8820_StringToEnum(commandArray, resBuf0, error)) < 0) */
   /*    return auMT8820_status;*/

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Call Processing
       Command(s): CALLPROC[?]
       Purpose:    Set/Get the call processing mode
       Mode defined by:  AUMT8820_CALL_PROCESSING_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Call_Processing(ViSession instrSession, ViInt32 mode)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"ON", "OFF", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   if (AuMT8820_InvalidViInt32Range(mode, 0, 1) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   len = sprintf(buf, "CALLPROC %s", commandArray[mode]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Call Processing
       Command(s): CALLPROC[?]
       Purpose:    Set/Get the call processing mode
       Mode defined by:  AUMT8820_CALL_PROCESSING_xxx
 */
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Call_Processing(ViSession instrSession, ViInt32  _VI_FAR *mode)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"ON", "OFF", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViChar    resBuf0[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "CALLPROC?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   /* Get if numeric returned from instrument */
   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   /* Scan(rdBuf, "%d", mode); */

   /* Get if string returned from instrument */
   Scan(rdBuf, "%s", resBuf0); 
   if((auMT8820_status = AuMT8820_StringToEnum(commandArray, resBuf0, mode)) < 0) 
     return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Refresh Call Status
       Command(s): CALLRFR
       Purpose:    Forcibly sets the call processing status to Idle regardless of current status
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Refresh_Call_Status_WCDMA(ViSession instrSession)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   len = sprintf(buf, "CALLRFR");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Call Processing Result WCDMA
       Command(s): CALLRSLT?
       Purpose:    Get the result for call processing specifying a sequence
       Code defined by:  AUMT8820_CALL_PROCESSING_WCDMA_xxx
       Flag defined by:  AUMT8820_CALL_PROCESSING_FLAG_xxx
 */
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Call_Processing_Result_WCDMA(ViSession instrSession, ViInt32 code, ViInt32 _VI_FAR *flag, ViInt32 _VI_FAR *error)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
//   static ViString comandArray[] = {"", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViChar    resBuf0[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   if (AuMT8820_InvalidViInt32Range(code, 0, 16) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   len = sprintf(buf, "CALLRSLT? %d", code);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   /* Get if numeric returned from instrument */
   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%d,%d", flag, error);

   /* Get if string returned from instrument */
   /*  Scan(rdBuf, "%s", resBuf0); */
   /*  if((auMT8820_status = AuMT8820_StringToEnum(commandArray, resBuf0, flag)) < 0) */
   /*    return auMT8820_status;*/

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Start Call
       Command(s): CALLSA
       Purpose:    To connect a call to a terminal
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Start_Call(ViSession instrSession)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   len = sprintf(buf, "CALLSA");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Stop Call
       Command(s): CALLSO
       Purpose:    To release a call
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Stop_Call(ViSession instrSession)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   len = sprintf(buf, "CALLSO");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Call Processing Status WCDMA
       Command(s): CALLSTAT?
       Purpose:    Get the current call processing status
       Status defined by:  AUMT8820_CALL_PROCESSING_WCDMA_xxx
 */
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Call_Processing_Status_WCDMA(ViSession instrSession, ViInt32 _VI_FAR *status)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   //static ViString comandArray[] = {"", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViChar    resBuf0[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "CALLSTAT?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   /* Get if numeric returned from instrument */
   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%d", status);

   /* Get if string returned from instrument */
   /*  Scan(rdBuf, "%s", resBuf0); */
   /*  if((auMT8820_status = AuMT8820_StringToEnum(commandArray, resBuf0, status)) < 0) */
   /*    return auMT8820_status;*/

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Call Threshold
       Command(s): CALLTHLD[?]
       Purpose:    Sets the threshhold at which a call may be dropped.
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Call_Threshold(ViSession instrSession, ViInt32 frame)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   if (AuMT8820_InvalidViInt32Range(frame,100,1024) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   len = sprintf(buf, "CALLTHLD %d", frame);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Call Threshold
       Command(s): CALLTHLD[?]
       Purpose:    Sets the threshhold at which a call may be dropped.
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Call_Threshold(ViSession instrSession,  ViInt32 _VI_FAR *frame)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "CALLTHLD?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%d", frame);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Carrier Frequency
       Command(s): CARRF?
       Purpose:    Gets the result for the carrier frequency measurement.
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Carrier_Frequency_WCDMA(ViSession instrSession, ViReal64 _VI_FAR *frequency_Hz)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "CARRF?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%f", frequency_Hz);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Carrier Frequency Error
       Command(s): CARRFERR?
       Purpose:    Get the carrier frequency error
       Unit defined by:  AUMT8820_CARRIER_FREQ_ERROR_UNIT_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Carrier_Frequency_Error_WCDMA(ViSession instrSession, ViInt32 unit, ViReal64 _VI_FAR *result)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"HZ", "PPM", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   if (AuMT8820_InvalidViUInt32Range(unit, 0, 1) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   len = sprintf(buf, "CARRFERR? %s", commandArray[unit]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%f", result);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Cell Identity
       Command(s): CELLID[?]
       Purpose:    Sets the cell identification code.
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Cell_Identity(ViSession instrSession, ViInt32 iD)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   if (AuMT8820_InvalidViInt32Range(iD,0x0,0x0FFFFFFF) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   len = sprintf(buf, "CELLID %X", iD);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Cell Identity
       Command(s): CELLID[?]
       Purpose:    Sets the cell identification code.
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Cell_Identity(ViSession instrSession,  ViChar _VI_FAR *iD)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "CELLID?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%s", iD);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Uplink Channel WCDMA
       Command(s): CHAN[?],ULCHAN[?]
       Purpose:    Get/Set the uplink channel number
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Uplink_Channel_WCDMA(ViSession instrSession, ViInt32 channel)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   if (AuMT8820_InvalidViInt32Range(channel,1,12550) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   len = sprintf(buf, "CHAN %d", channel);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Uplink Channel WCDMA
       Command(s): CHAN[?],ULCHAN[?]
       Purpose:    Get/Set the uplink channel number
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Uplink_Channel_WCDMA(ViSession instrSession,  ViInt32 _VI_FAR *channel)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "CHAN?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%d", channel);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Channel Coding Object
       Command(s): CHCODING[?]
       Purpose:    Set/Get the channel coding object
       Mode defined by:  AUMT8820_CHANNEL_CODING_OBJECT_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Channel_Coding_Object(ViSession instrSession, ViInt32 mode)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"REFMEASCH", "VOICE", "AV", "PACKET", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   if (AuMT8820_InvalidViInt32Range(mode, 0, 3) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   len = sprintf(buf, "CHCODING %s", commandArray[mode]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Channel Coding Object
       Command(s): CHCODING[?]
       Purpose:    Set/Get the channel coding object
       Mode defined by:  AUMT8820_CHANNEL_CODING_OBJECT_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Channel_Coding_Object(ViSession instrSession, ViInt32  _VI_FAR *mode)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"REFMEASCH", "VOICE", "AV", "PACKET", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViChar    resBuf0[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "CHCODING?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   /* Get if numeric returned from instrument */
   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   /* Scan(rdBuf, "%d", mode); */

   /* Get if string returned from instrument */
   Scan(rdBuf, "%s", resBuf0); 
   if((auMT8820_status = AuMT8820_StringToEnum(commandArray, resBuf0, mode)) < 0) 
     return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set AICH Condition of Returning Acknowledge
       Command(s): CONDIACK[?]
       Purpose:    Sets the AICH to On or Off
       Setting defined by:  AUMT8820_AICH_CONDITION_OF_RETURNING_ACKNOWLEDGE_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_AICH_Condition_of_Returning_Acknowledge(ViSession instrSession, ViInt32 setting)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"ON", "OFF", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   if (AuMT8820_InvalidViInt32Range(setting, 0, 1) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   len = sprintf(buf, "CONDIACK %s", commandArray[setting]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get AICH Condition of Returning Acknowledge
       Command(s): CONDIACK[?]
       Purpose:    Sets the AICH to On or Off
       Setting defined by:  AUMT8820_AICH_CONDITION_OF_RETURNING_ACKNOWLEDGE_xxx
 */
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_AICH_Condition_of_Returning_Acknowledge(ViSession instrSession, ViInt32  _VI_FAR *setting)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"ON", "OFF", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViChar    resBuf0[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "CONDIACK?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   /* Get if numeric returned from instrument */
   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   /* Scan(rdBuf, "%d", setting); */

   /* Get if string returned from instrument */
   Scan(rdBuf, "%s", resBuf0); 
   if((auMT8820_status = AuMT8820_StringToEnum(commandArray, resBuf0, setting)) < 0) 
     return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set RACH Constant Value
       Command(s): CONSTANT[?]
       Purpose:    Sets the constant value for RACH parameter.
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_RACH_Constant_Value(ViSession instrSession, ViInt32 level_dB)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   len = sprintf(buf, "CONSTANT %d", level_dB);
   if (AuMT8820_InvalidViInt32Range(level_dB, -35, 10) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get RACH Constant Value
       Command(s): CONSTANT[?]
       Purpose:    Sets the constant value for RACH parameter.
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_RACH_Constant_Value(ViSession instrSession, ViInt32 _VI_FAR *level_dB)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "CONSTANT?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%d", level_dB);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Continuous Mode Measure
       Command(s): CONTS,S1
       Purpose:    Starts measurement in continuous mode.
       Defined in MainFrame LabWindows Driver.c
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Continuous_Mode_Measure(ViSession instrSession)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   len = sprintf(buf, "CONTS");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set CPICH Downlink
       Command(s): CPICH[?]
       Purpose:    Sets/Gets the CPICH output for downlink physical channel to On or Off.
       Setting defined by:  AUMT8820_CPICH_DOWNLINK_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_CPICH_Downlink(ViSession instrSession, ViInt32 setting)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"ON", "OFF", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   if (AuMT8820_InvalidViInt32Range(setting, 0, 1) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   len = sprintf(buf, "CPICH %s", commandArray[setting]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get CPICH Downlink
       Command(s): CPICH[?]
       Purpose:    Sets/Gets the CPICH output for downlink physical channel to On or Off.
       Setting defined by:  AUMT8820_CPICH_DOWNLINK_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_CPICH_Downlink(ViSession instrSession, ViInt32  _VI_FAR *setting)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"ON", "OFF", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViChar    resBuf0[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "CPICH?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   /* Get if numeric returned from instrument */
   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   /* Scan(rdBuf, "%d", setting); */

   /* Get if string returned from instrument */
   Scan(rdBuf, "%s", resBuf0); 
   if((auMT8820_status = AuMT8820_StringToEnum(commandArray, resBuf0, setting)) < 0) 
     return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get CPICH Power Ratio
       Command(s): CPICH_ECN0?
       Purpose:    Queries the power ratio between CPICH and noise.
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_CPICH_Power_Ratio(ViSession instrSession, ViInt32 _VI_FAR *ratio_dB)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "CPICH_ECN0?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%d", ratio_dB);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set CPICH Downlink Power
       Command(s): CPICHPWR[?]
       Purpose:    Sets the downlink CPICH output level.
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_CPICH_Downlink_Power(ViSession instrSession, ViReal64 range_dB)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   len = sprintf(buf, "CPICHPWR %.1f", range_dB);
   if (AuMT8820_InvalidViReal64Range(range_dB, -30.000000, 0.000000) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get CPICH Downlink Power
       Command(s): CPICHPWR[?]
       Purpose:    Sets the downlink CPICH output level.
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_CPICH_Downlink_Power(ViSession instrSession, ViReal64 _VI_FAR *range_dB)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "CPICHPWR?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%f", range_dB);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get CPICH Power Received
       Command(s): CPICH_RSCP?
       Purpose:    Queries the power received by CPICH after demodulation.
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_CPICH_Power_Received(ViSession instrSession, ViInt32 _VI_FAR *level_dB)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "CPICH_RSCP?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%d", level_dB);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set CPICH Primary TX Power
       Command(s): CPICHTXPWR[?]
       Purpose:    Sets the transmitted power for CPICH primary download.
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_CPICH_Primary_TX_Power(ViSession instrSession, ViInt32 range_dBm)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   len = sprintf(buf, "CPICHTXPWR %dDBM", range_dBm);
   if (AuMT8820_InvalidViInt32Range(range_dBm, -10, 50) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get CPICH Primary TX Power
       Command(s): CPICHTXPWR[?]
       Purpose:    Sets the transmitted power for CPICH primary download.
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_CPICH_Primary_TX_Power(ViSession instrSession, ViInt32 _VI_FAR *range_dBm)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "CPICHTXPWR?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%d", range_dBm);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get DCCH Signaling
       Command(s): DCCH?
       Purpose:    Queries the DCCH transmission rate for signaling.
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_DCCH_Signaling(ViSession instrSession, ViReal64 _VI_FAR *rate)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "DCCH?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%f", rate);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set DCCH Data Pattern
       Command(s): DCCHPAT[?]
       Purpose:    Sets/Gets the DCCH data pattern.
       Pattern defined by:  AUMT8820_DCCH_DATA_PATTERN_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_DCCH_Data_Pattern(ViSession instrSession, ViInt32 pattern)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"ALL0", "ALL1", "PN9", "PN15", "SIGNAL", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   if (AuMT8820_InvalidViInt32Range(pattern, 0, 4) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   len = sprintf(buf, "DCCHPAT %s", commandArray[pattern]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get DCCH Data Pattern
       Command(s): DCCHPAT[?]
       Purpose:    Sets/Gets the DCCH data pattern.
       Pattern defined by:  AUMT8820_DCCH_DATA_PATTERN_xxx
 */
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_DCCH_Data_Pattern(ViSession instrSession, ViInt32  _VI_FAR *pattern)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"ALL0", "ALL1", "PN9", "PN15", "SIGNAL", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViChar    resBuf0[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "DCCHPAT?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   /* Get if numeric returned from instrument */
   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   /* Scan(rdBuf, "%d", pattern); */

   /* Get if string returned from instrument */
   Scan(rdBuf, "%s", resBuf0); 
   if((auMT8820_status = AuMT8820_StringToEnum(commandArray, resBuf0, pattern)) < 0) 
     return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set DPCH Downlink Channel
       Command(s): DDPCH[?]
       Purpose:    Sets/Gets the DCCH downlink channel to On or Off.
       Setting defined by:  AUMT8820_DPCH_DOWNLINK_CHANNEL_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_DPCH_Downlink_Channel(ViSession instrSession, ViInt32 setting)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"ON", "OFF", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   if (AuMT8820_InvalidViInt32Range(setting, 0, 1) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   len = sprintf(buf, "DDPCH %s", commandArray[setting]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get DPCH Downlink Channel
       Command(s): DDPCH[?]
       Purpose:    Sets/Gets the DCCH downlink channel to On or Off.
       Setting defined by:  AUMT8820_DPCH_DOWNLINK_CHANNEL_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_DPCH_Downlink_Channel(ViSession instrSession, ViInt32  _VI_FAR *setting)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"ON", "OFF", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViChar    resBuf0[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "DDPCH?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   /* Get if numeric returned from instrument */
   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   /* Scan(rdBuf, "%d", setting); */

   /* Get if string returned from instrument */
   Scan(rdBuf, "%s", resBuf0); 
   if((auMT8820_status = AuMT8820_StringToEnum(commandArray, resBuf0, setting)) < 0) 
     return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set RAB Channelization Code
       Command(s): DDPCHCODE[?]
       Purpose:    Sets the channelization code for RAB.
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_RAB_Channelization_Code(ViSession instrSession, ViInt32 range)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   if (AuMT8820_InvalidViInt32Range(range,0,127) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   len = sprintf(buf, "DDPCHCODE %d", range);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get RAB Channelization Code
       Command(s): DDPCHCODE[?]
       Purpose:    Sets the channelization code for RAB.
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_RAB_Channelization_Code(ViSession instrSession,  ViInt32 _VI_FAR *range)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "DDPCHCODE?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%d", range);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set DPCH Downlink Power
       Command(s): DDPCHPWR[?]
       Purpose:    Sets the downlink DPCH output level.
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_DPCH_Downlink_Power(ViSession instrSession, ViReal64 range_dB)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   len = sprintf(buf, "DDPCHPWR %.1f", range_dB);
   if (AuMT8820_InvalidViReal64Range(range_dB, -30.000000, 0.000000) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get DPCH Downlink Power
       Command(s): DDPCHPWR[?]
       Purpose:    Sets the downlink DPCH output level.
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_DPCH_Downlink_Power(ViSession instrSession, ViReal64 _VI_FAR *range_dB)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "DDPCHPWR?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%f", range_dB);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get DPCH Downlink Symbol Rate
       Command(s): DDPCHRATE?
       Purpose:    Queries the symbol rate for downlink DPCH.
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_DPCH_Downlink_Symbol_Rate(ViSession instrSession, ViInt32 _VI_FAR *code)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "DDPCHRATE?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%d", code);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set DCCH Channelization Code
       Command(s): DDPCHSACODE[?]
       Purpose:    Sets the channelization code for stand alone DCCH.
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_DCCH_Channelization_Code(ViSession instrSession, ViInt32 code)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   if (AuMT8820_InvalidViInt32Range(code,0,127) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   len = sprintf(buf, "DDPCHSACODE %d", code);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get DCCH Channelization Code
       Command(s): DDPCHSACODE[?]
       Purpose:    Sets the channelization code for stand alone DCCH.
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_DCCH_Channelization_Code(ViSession instrSession,  ViInt32 _VI_FAR *code)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "DDPCHSACODE?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%d", code);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set DCCH Secondary Scrambling Code
       Command(s): DDPCHSASSCRCODE[?]
       Purpose:    Sets the secondary scrambling code for stand alone DCCH.
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_DCCH_Secondary_Scrambling_Code(ViSession instrSession, ViInt32 code)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   if (AuMT8820_InvalidViInt32Range(code,0,15) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   len = sprintf(buf, "DDPCHSASSCRCODE %d", code);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get DCCH Secondary Scrambling Code
       Command(s): DDPCHSASSCRCODE[?]
       Purpose:    Sets the secondary scrambling code for stand alone DCCH.
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_DCCH_Secondary_Scrambling_Code(ViSession instrSession,  ViInt32 _VI_FAR *code)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "DDPCHSASSCRCODE?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%d", code);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set DPCH Secondary Scrambling Code
       Command(s): DDPCHSSCRCODE[?]
       Purpose:    Sets the secondary scrambling code for stand alone DPCH.
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_DPCH_Secondary_Scrambling_Code(ViSession instrSession, ViInt32 code)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   if (AuMT8820_InvalidViInt32Range(code,0,15) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   len = sprintf(buf, "DDPCHSSCRCODE %d", code);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get DPCH Secondary Scrambling Code
       Command(s): DDPCHSSCRCODE[?]
       Purpose:    Sets the secondary scrambling code for stand alone DPCH.
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_DPCH_Secondary_Scrambling_Code(ViSession instrSession,  ViInt32 _VI_FAR *code)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "DDPCHSSCRCODE?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%d", code);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set DPCH Timing Offset
       Command(s): DDPCHTOFS[?]
       Purpose:    Sets the timing offset for downlink DPCH.
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_DPCH_Timing_Offset(ViSession instrSession, ViInt32 offset)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   if (AuMT8820_InvalidViInt32Range(offset,0,149) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   len = sprintf(buf, "DDPCHTOFS %d", offset);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get DPCH Timing Offset
       Command(s): DDPCHTOFS[?]
       Purpose:    Sets the timing offset for downlink DPCH.
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_DPCH_Timing_Offset(ViSession instrSession,  ViInt32 _VI_FAR *offset)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "DDPCHTOFS?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%d", offset);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Delete Slot from List
       Command(s): DELSLOTLIST
       Purpose:    Deletes slot numbers from the slot power window of the time domain measurement screen.
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Delete_Slot_from_List(ViSession instrSession, ViInt32 slot)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   if (AuMT8820_InvalidViInt32Range(slot,-15,164) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   len = sprintf(buf, "DELSLOTLIST %d", slot);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}

/*****************************************************************************/
/*
       Function:   Delete Slot from List Range
       Command(s): DELSLOTLIST n-m
       Purpose:    Deletes slot numbers from the slot power window of the time domain measurement screen.
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Delete_Slot_from_List_Range(ViSession instrSession, ViInt32 slotn, ViInt32 slotm)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   if (AuMT8820_InvalidViInt32Range(slotn,-15,164) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   if (AuMT8820_InvalidViInt32Range(slotm,-15,164) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   if (slotn >= slotm)
      return VI_ERROR_OUTOF_RANGE;
   len = sprintf(buf, "DELSLOTLIST %d-%d", slotn, slotm);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Downlink Channel
       Command(s): DLCHAN[?]
       Purpose:    Sets the downlink channel.
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Downlink_Channel(ViSession instrSession, ViInt32 channel)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   if (AuMT8820_InvalidViInt32Range(channel,951,13500) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   len = sprintf(buf, "DLCHAN %d", channel);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Downlink Channel
       Command(s): DLCHAN[?]
       Purpose:    Sets the downlink channel.
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Downlink_Channel(ViSession instrSession,  ViInt32 _VI_FAR *channel)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "DLCHAN?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%d", channel);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Ior Total Power
       Command(s): DLCHPWR_IOR?
       Purpose:    Queries the relative value between the output power and total level for each physical channel.
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Ior_Total_Power(ViSession instrSession, ViReal64 _VI_FAR *level_dB)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "DLCHPWR_IOR?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%f", level_dB);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set External Loss On Off
       Command(s): DLEXTLOSSW[?]
       Purpose:    Sets/Gets the external loss at the main I/O to On or Off.
       Mode defined by:  AUMT8820_EXTERNAL_LOSS_MODE_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_External_Loss_On_Off(ViSession instrSession, ViInt32 mode)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"ON", "OFF", "COMMON", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   if (AuMT8820_InvalidViInt32Range(mode, 0, 2) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   len = sprintf(buf, "DLEXTLOSSW %s", commandArray[mode]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get External Loss On Off
       Command(s): DLEXTLOSSW[?]
       Purpose:    Sets/Gets the external loss at the main I/O to On or Off.
       Mode defined by:  AUMT8820_EXTERNAL_LOSS_MODE_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_External_Loss_On_Off(ViSession instrSession, ViInt32  _VI_FAR *mode)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"ON", "OFF", "COMMON", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViChar    resBuf0[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "DLEXTLOSSW?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   /* Get if numeric returned from instrument */
   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   /* Scan(rdBuf, "%d", mode); */

   /* Get if string returned from instrument */
   Scan(rdBuf, "%s", resBuf0); 
   if((auMT8820_status = AuMT8820_StringToEnum(commandArray, resBuf0, mode)) < 0) 
     return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set External Down Link Loss WCDMA
       Command(s): DLEXTLOSS[?]
       Purpose:    Sets the external loss at the main I/O downlink
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_External_Down_Link_Loss_WCDMA(ViSession instrSession, ViReal64 level_dB)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   len = sprintf(buf, "DLEXTLOSS %.1f", level_dB);
   if (AuMT8820_InvalidViReal64Range(level_dB, -55.000000, 55.000000) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get External Down Link Loss WCDMA
       Command(s): DLEXTLOSS[?]
       Purpose:    Sets the external loss at the main I/O downlink
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_External_Down_Link_Loss_WCDMA(ViSession instrSession, ViReal64 _VI_FAR *level_dB)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "DLEXTLOSS?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%f", level_dB);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Downlink Frequency
       Command(s): DLFREQ[?],RFREQ[?],RXFREQ[?]
       Purpose:    Get/Set the downlink frequency (resolution 1 Hz)
       Units defined by:  AUMT8820_FREQUENCY_UNIT_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Downlink_Frequency(ViSession instrSession, ViReal64 freq, ViInt32 units)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString unitString[] = {"HZ", "KHZ", "MHZ", "GHZ", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   if (AuMT8820_InvalidViReal64Range(freq, 0.0004, 2700000000.00) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   if (AuMT8820_InvalidViInt32Range(units, 0, 3) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   len = sprintf(buf, "DLFREQ %.12f%s", freq, unitString[units]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Downlink Frequency
       Command(s): DLFREQ[?],RFREQ[?],RXFREQ[?]
       Purpose:    Get/Set the downlink frequency (resolution 1 Hz)
       Units defined by:  AUMT8820_FREQUENCY_UNIT_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Downlink_Frequency(ViSession instrSession, ViReal64 _VI_FAR *freq)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString unitString[] = {"HZ", "KHZ", "MHZ", "GHZ", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViChar    resBuf0[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "DLFREQ?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%f%s", freq, resBuf0);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Digital Traffic Channel Data Pattern
       Command(s): DTCHPAT[?]
       Purpose:    Set/Get the digital traffic channel (DTCH) data pattern
       Pattern defined by:  AUMT8820_DIGITAL_TRAFFIC_CHANNEL_DATA_PATTERN_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Digital_Traffic_Channel_Data_Pattern(ViSession instrSession, ViInt32 pattern)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"ALL0", "ALL1", "PN9", "PN15", "ECHO", "VOICE", "NODATA", "EPACKETDT", "EIPPACKET", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   if (AuMT8820_InvalidViInt32Range(pattern, 0, 8) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   len = sprintf(buf, "DTCHPAT %s", commandArray[pattern]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Digital Traffic Channel Data Pattern
       Command(s): DTCHPAT[?]
       Purpose:    Set/Get the digital traffic channel (DTCH) data pattern
       Pattern defined by:  AUMT8820_DIGITAL_TRAFFIC_CHANNEL_DATA_PATTERN_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Digital_Traffic_Channel_Data_Pattern(ViSession instrSession, ViInt32  _VI_FAR *pattern)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"ALL0", "ALL1", "PN9", "PN15", "ECHO", "VOICE", "NODATA", "EPACKETDT", "EIPPACKET", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViChar    resBuf0[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "DTCHPAT?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   /* Get if numeric returned from instrument */
   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   /* Scan(rdBuf, "%d", pattern); */

   /* Get if string returned from instrument */
   Scan(rdBuf, "%s", resBuf0); 
   if((auMT8820_status = AuMT8820_StringToEnum(commandArray, resBuf0, pattern)) < 0) 
     return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Error Vector Magnitude
       Command(s): EVM?
       Purpose:    Get the error vector magnitude in percent
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Error_Vector_Magnitude(ViSession instrSession, ViReal64 _VI_FAR *percent)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "EVM?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%f", percent);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Filtered Power
       Command(s): FILTPWR?
       Purpose:    Queries the most recent measured result of RCC filtered transmission power in the fundamental measurement screen.
       Unit defined by:  AUMT8820_FILTERED_POWER_UNIT_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Filtered_Power(ViSession instrSession, ViInt32 units, ViReal64 _VI_FAR *power)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"DBM", "WATT", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "FILTPWR? %s", commandArray[units]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%f", power);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Frequency Separation
       Command(s): FREQSEP
       Purpose:    Sets the frequency separation between uplink and downlink.
*/
/*****************************************************************************/

ViStatus _VI_FUNC AuMT8820_Set_Frequency_Separation(ViSession instrSession, ViReal64 frequency_MHz)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   len = sprintf(buf, "FREQSEP %.1fMHZ", frequency_MHz);
   if (AuMT8820_InvalidViReal64Range(frequency_MHz, 60.0, 245.2) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}



/*****************************************************************************/
/*
       Function:   Get Frequency Separation
       Command(s): FREQSEP?
       Purpose:    Queries the frequency separation between uplink and downlink.
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Frequency_Separation(ViSession instrSession, ViReal64 _VI_FAR *frequency_Hz)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "FREQSEP?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%f", frequency_Hz);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Average Count for Frequency Error Measurement
       Command(s): FREQ_AVG[?]
       Purpose:    Set/Get the average count for frequency error measurements
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Average_Count_for_Frequency_Error_Measurement(ViSession instrSession, ViInt32 count)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   if (AuMT8820_InvalidViInt32Range(count,1,9999) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   len = sprintf(buf, "FREQ_AVG %d", count);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Average Count for Frequency Error Measurement
       Command(s): FREQ_AVG[?]
       Purpose:    Set/Get the average count for frequency error measurements
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Average_Count_for_Frequency_Error_Measurement(ViSession instrSession,  ViInt32 _VI_FAR *count)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "FREQ_AVG?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%d", count);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Frequency Error Measurement Mode
       Command(s): FREQ_MEAS[?]
       Purpose:    Set/Get the frequency error measurement mode
       Mode defined by:  AUMT8820_FREQUENCY_ERROR_MEASUREMENT_MODE_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Frequency_Error_Measurement_Mode(ViSession instrSession, ViInt32 mode)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"ON", "OFF", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   if (AuMT8820_InvalidViInt32Range(mode, 0, 1) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   len = sprintf(buf, "FREQ_MEAS %s", commandArray[mode]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Frequency Error Measurement Mode
       Command(s): FREQ_MEAS[?]
       Purpose:    Set/Get the frequency error measurement mode
       Mode defined by:  AUMT8820_FREQUENCY_ERROR_MEASUREMENT_MODE_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Frequency_Error_Measurement_Mode(ViSession instrSession, ViInt32  _VI_FAR *mode)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"ON", "OFF", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViChar    resBuf0[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "FREQ_MEAS?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   /* Get if numeric returned from instrument */
   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   /* Scan(rdBuf, "%d", mode); */

   /* Get if string returned from instrument */
   Scan(rdBuf, "%s", resBuf0); 
   if((auMT8820_status = AuMT8820_StringToEnum(commandArray, resBuf0, mode)) < 0) 
     return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Full Calibration
       Command(s): FULLCALL
       Purpose:    Calibration for all bands
       Defined in MT8820 GSM LabWindows Driver.c
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Full_Calibration_WCDMA(ViSession instrSession)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   len = sprintf(buf, "FULLCALL");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set TPC Command
       Command(s): ILP_CMD[?]
       Purpose:    Sets the TPC command at step A.
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_TPC_Command(ViSession instrSession, ViInt32 term, ViChar *bit)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   if (AuMT8820_InvalidViInt32Range(term,1,12) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
//   if (AuMT8820_InvalidViInt32Range(bit,0,31) == VI_TRUE)
//      return VI_ERROR_OUTOF_RANGE;
   len = sprintf(buf, "ILP_CMD %d,%s", term, bit);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get TPC Command
       Command(s): ILP_CMD[?]
       Purpose:    Sets the TPC command at step A.
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_TPC_Command(ViSession instrSession, ViInt32 term,  ViChar _VI_FAR *bit)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   if (AuMT8820_InvalidViInt32Range(term,1,12) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   len = sprintf(buf, "ILP_CMD? %d", term);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%s", bit);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set TPC Command All
       Command(s): ILP_CMDALL[?]
       Purpose:    Sets/Gets all the TPC command at step A.
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_TPC_Command_All(ViSession instrSession, ViChar *bit)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
/*   if (AuMT8820_InvalidViInt32Range(term[0],0,31) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   if (AuMT8820_InvalidViInt32Range(term[1],0,31) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   if (AuMT8820_InvalidViInt32Range(term[2],0,31) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   if (AuMT8820_InvalidViInt32Range(term[3],0,31) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   if (AuMT8820_InvalidViInt32Range(term[4],0,31) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   if (AuMT8820_InvalidViInt32Range(term[5],0,31) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   if (AuMT8820_InvalidViInt32Range(term[6],0,31) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   if (AuMT8820_InvalidViInt32Range(term[7],0,31) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   if (AuMT8820_InvalidViInt32Range(term[8],0,31) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   if (AuMT8820_InvalidViInt32Range(term[9],0,31) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   if (AuMT8820_InvalidViInt32Range(term[10],0,31) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   if (AuMT8820_InvalidViInt32Range(term[11],0,31) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;

   len = sprintf(buf, "ILP_CMDALL %05b,%05b,%05b,%05b,%05b,%05b,%05b,%05b,%05b,%05b,%05b,%05b", */
   len = sprintf(buf, "ILP_CMDALL %s",bit);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get TPC Command All
       Command(s): ILP_CMDALL[?]
       Purpose:    Sets/Gets all the TPC command at step A.
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_TPC_Command_All(ViSession instrSession, ViChar _VI_FAR *bit)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "ILP_CMDALL?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%s",bit);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set TPC Command Length
       Command(s): ILP_CMDSLOT[?]
       Purpose:    Sets/Gets the length ofTPC command when TPC method is set to step B to H.
       Method defined by:  AUMT8820_TPC_COMMAND_TYPE_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_TPC_Command_Length(ViSession instrSession, ViInt32 method, ViInt32 length)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"B", "C", "D", "E", "F", "G", "H", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   if (AuMT8820_InvalidViUInt32Range(method, 0, 6) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   if (AuMT8820_InvalidViInt32Range(length, 1, 150) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   len = sprintf(buf, "ILP_CMDSLOT %s,%d", commandArray[method], length);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get TPC Command Length
       Command(s): ILP_CMDSLOT[?]
       Purpose:    Sets/Gets the length ofTPC command when TPC method is set to step B to H.
       Method defined by:  AUMT8820_TPC_COMMAND_TYPE_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_TPC_Command_Length(ViSession instrSession, ViInt32 method,  ViInt32 _VI_FAR *length)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"B", "C", "D", "E", "F", "G", "H", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "ILP_CMDSLOT? %s", commandArray[method]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%d", length);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set TPC Command Type
       Command(s): ILP_CMDTYPE[?]
       Purpose:    Sets/Gets the TPC method to step B to H.
       Method defined by:  AUMT8820_TPC_COMMAND_TYPE_xxx
       Type defined by:  AUMT8820_TPC_COMMAND_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_TPC_Command_Type(ViSession instrSession, ViInt32 method, ViInt32 type)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"B", "C", "D", "E", "F", "G", "H", 0L};
   static ViString commandArray2[] = {"ALL0", "ALL1", "ALT", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   if (AuMT8820_InvalidViUInt32Range(method, 0, 6) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   if (AuMT8820_InvalidViInt32Range(type, 0, 2) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   len = sprintf(buf, "ILP_CMDTYPE %s,%s", commandArray[method], commandArray2[type]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get TPC Command Type
       Command(s): ILP_CMDTYPE[?]
       Purpose:    Sets/Gets the TPC method to step B to H.
       Method defined by:  AUMT8820_TPC_COMMAND_TYPE_xxx
       Type defined by:  AUMT8820_TPC_COMMAND_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_TPC_Command_Type(ViSession instrSession, ViInt32 method, ViInt32  _VI_FAR *type)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"B", "C", "D", "E", "F", "G", "H", 0L};
   static ViString commandArray2[] = {"ALL0", "ALL1", "ALT", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViChar    resBuf0[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "ILP_CMDTYPE? %s", commandArray[method]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   /* Get if numeric returned from instrument */
   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   /* Scan(rdBuf, "%d", type); */

   /* Get if string returned from instrument */
   Scan(rdBuf, "%s", resBuf0); 
   if((auMT8820_status = AuMT8820_StringToEnum(commandArray2, resBuf0, type)) < 0) 
     return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set TPC Command Mode
       Command(s): ILPC_STEPAMODE[?]
       Purpose:    Sets/Gets the TPC method to default or user defined.
       Mode defined by:  AUMT8820_TPC_COMMAND_MODE_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_TPC_Command_Mode(ViSession instrSession, ViInt32 mode)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"DEF", "USER", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   if (AuMT8820_InvalidViInt32Range(mode, 0, 1) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   len = sprintf(buf, "ILPC_STEPAMODE %s", commandArray[mode]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get TPC Command Mode
       Command(s): ILPC_STEPAMODE[?]
       Purpose:    Sets/Gets the TPC method to default or user defined.
       Mode defined by:  AUMT8820_TPC_COMMAND_MODE_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_TPC_Command_Mode(ViSession instrSession, ViInt32  _VI_FAR *mode)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"DEF", "USER", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViChar    resBuf0[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "ILPC_STEPAMODE?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   /* Get if numeric returned from instrument */
   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   /* Scan(rdBuf, "%d", mode); */

   /* Get if string returned from instrument */
   Scan(rdBuf, "%s", resBuf0); 
   if((auMT8820_status = AuMT8820_StringToEnum(commandArray, resBuf0, mode)) < 0) 
     return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set TPC Test Step
       Command(s): ILP_TPC[?]
       Purpose:    Sets/Gets the TPC test step.
       Step defined by:  AUMT8820_TPC_TEST_STEP_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_TPC_Test_Step(ViSession instrSession, ViInt32 step)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"A", "B", "C", "D", "E", "F", "G", "H", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   if (AuMT8820_InvalidViInt32Range(step, 0, 7) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   len = sprintf(buf, "ILP_TPC %s", commandArray[step]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get TPC Test Step
       Command(s): ILP_TPC[?]
       Purpose:    Sets/Gets the TPC test step.
       Step defined by:  AUMT8820_TPC_TEST_STEP_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_TPC_Test_Step(ViSession instrSession, ViInt32  _VI_FAR *step)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"A", "B", "C", "D", "E", "F", "G", "H", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViChar    resBuf0[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "ILP_TPC?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   /* Get if numeric returned from instrument */
   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   /* Scan(rdBuf, "%d", step); */

   /* Get if string returned from instrument */
   Scan(rdBuf, "%s", resBuf0); 
   if((auMT8820_status = AuMT8820_StringToEnum(commandArray, resBuf0, step)) < 0) 
     return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Int Mobile Sub Identity
       Command(s): IMSI[?]
       Purpose:    Sets/Gets the IMSI (International Mobile Subscriber Identity)
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Int_Mobile_Sub_Identity_WCDMA(ViSession instrSession, ViChar _VI_FAR iD[])
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   //if (AuMT8820_InvalidViInt32Range(iD,0,2147483647) == VI_TRUE)
   //   return VI_ERROR_OUTOF_RANGE;
   len = sprintf(buf, "IMSI %s", iD);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Int Mobile Sub Identity
       Command(s): IMSI[?]
       Purpose:    Sets/Gets the IMSI (International Mobile Subscriber Identity)
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Int_Mobile_Sub_Identity_WCDMA(ViSession instrSession,  ViChar _VI_FAR iD[])
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "IMSI?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%s", iD);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Uplink Interference
       Command(s): INTERFERENCE[?]
       Purpose:    Sets/Gets the interference power for the uplink.
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Uplink_Interference(ViSession instrSession, ViInt32 level_dB)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   len = sprintf(buf, "INTERFERENCE %dDB", level_dB);
   if (AuMT8820_InvalidViInt32Range(level_dB, -110, -70) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Uplink Interference
       Command(s): INTERFERENCE[?]
       Purpose:    Sets/Gets the interference power for the uplink.
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Uplink_Interference(ViSession instrSession, ViInt32 _VI_FAR *level_dB)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "INTERFERENCE?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%d", level_dB);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get IQ Imbalance
       Command(s): IQIMB?
       Purpose:    Get the IQ imbalance
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_IQ_Imbalance(ViSession instrSession, ViReal64 _VI_FAR *imbalance)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "IQIMB?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%f", imbalance);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Parameter Items Displayed WCDMA
       Command(s): ITEMLIST[?]
       Purpose:    Sets/Gets the number of parameter items to be displayed for each tag in the setting window.
       Tag defined by:  AUMT8820_PARAMETER_ITEMS_DISPLAYED_WCDMA_xxx
       Option defined by:  AUMT8820_PARAMETER_OPTION_WCDMA_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Parameter_Items_Displayed_WCDMA(ViSession instrSession, ViInt32 tag, ViInt32 option)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"COM", "PHYCH", "CALL", "TXMEAS", "RXMEAS", "FMEAS", "AUDIO", "TDMEAS", "SPMON", 0L};
   static ViString commandArray2[] = {"STD", "DETAIL", "NONDSPL", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   if (AuMT8820_InvalidViInt32Range(tag, 0, 8) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   if (AuMT8820_InvalidViInt32Range(option, 0, 2) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   len = sprintf(buf, "ITEMLIST %s,%s", commandArray[tag], commandArray2[option]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Parameter Items Displayed WCDMA
       Command(s): ITEMLIST[?]
       Purpose:    Sets/Gets the number of parameter items to be displayed for each tag in the setting window.
       Tag defined by:  AUMT8820_PARAMETER_ITEMS_DISPLAYED_WCDMA_xxx
       Option defined by:  AUMT8820_PARAMETER_OPTION_WCDMA_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Parameter_Items_Displayed_WCDMA(ViSession instrSession, ViInt32 tag, ViInt32  _VI_FAR *option)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"COM", "PHYCH", "CALL", "TXMEAS", "RXMEAS", "FMEAS", "AUDIO", "TDMEAS", "SPMON",  0L};
   static ViString commandArray2[] = {"STD", "DETAIL", "NONDSPL", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViChar    resBuf0[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "ITEMLIST? %s", commandArray[tag]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   /* Get if numeric returned from instrument */
   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   /* Scan(rdBuf, "%d", option);  */

   /* Get if string returned from instrument */
   Scan(rdBuf, "%s", resBuf0); 
   if((auMT8820_status = AuMT8820_StringToEnum(commandArray2, resBuf0, option)) < 0) 
     return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Location ID
       Command(s): LAC[?]
       Purpose:    Sets/Gets the LAC for location area identification.
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Location_ID_WCDMA(ViSession instrSession, ViInt32 iD)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   if (AuMT8820_InvalidViInt32Range(iD,0x0,0x0FFFF) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   len = sprintf(buf, "LAC %0X", iD);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Location ID
       Command(s): LAC[?]
       Purpose:    Sets/Gets the LAC for location area identification.
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Location_ID_WCDMA(ViSession instrSession,  ViChar iD[])
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "LAC?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%X", iD);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Set Cursor
       Command(s): LISTCSR_SET[?]
       Purpose:    Sets/Gets the registered slot in which the cursor resides.
       NOT FUNCTIONAL
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Set_Cursor(ViSession instrSession, ViInt32 slot)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   if (AuMT8820_InvalidViInt32Range(slot,-15,164) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   len = sprintf(buf, "LISTCSR_SET %d", slot);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Set Cursor
       Command(s): LISTCSR_SET[?]
       Purpose:    Sets/Gets the registered slot in which the cursor resides.
       NOT FUNCTIONAL
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Set_Cursor(ViSession instrSession,  ViInt32 _VI_FAR *slot)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "LISTCSR_SET?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%d", slot);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Spectrum Emission Mask Template Lower Limit
       Command(s): LMTLVL_SMASK[?]
       Purpose:    Sets/Gets the lower limit for the spectrum emission mask template of the fundamental measurement screen when a signal is input at LPF= 3.84 MHz.
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Spectrum_Emission_Mask_Template_Lower_Limit(ViSession instrSession, ViReal64 level_dBm)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   len = sprintf(buf, "LMTLVL_SMASK %.1f", level_dBm);
   if (AuMT8820_InvalidViReal64Range(level_dBm, -100.000000, 0.000000) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Spectrum Emission Mask Template Lower Limit
       Command(s): LMTLVL_SMASK[?]
       Purpose:    Sets/Gets the lower limit for the spectrum emission mask template of the fundamental measurement screen when a signal is input at LPF= 3.84 MHz.
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Spectrum_Emission_Mask_Template_Lower_Limit(ViSession instrSession, ViReal64 _VI_FAR *level_dBm)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "LMTLVL_SMASK?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%f", level_dBm);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Long Span Code Search
       Command(s): LSCODESEARCH[?]
       Purpose:    Sets/Gets the long span code search to On or Off.
       Setting defined by:  AUMT8820_LONG_SPAN_CODE_SEARCH_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Long_Span_Code_Search(ViSession instrSession, ViInt32 setting)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"ON", "OFF", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   if (AuMT8820_InvalidViInt32Range(setting, 0, 1) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   len = sprintf(buf, "LSCODESEARCH %s", commandArray[setting]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Long Span Code Search
       Command(s): LSCODESEARCH[?]
       Purpose:    Sets/Gets the long span code search to On or Off.
       Setting defined by:  AUMT8820_LONG_SPAN_CODE_SEARCH_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Long_Span_Code_Search(ViSession instrSession, ViInt32  _VI_FAR *setting)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"ON", "OFF", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViChar    resBuf0[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "LSCODESEARCH?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   /* Get if numeric returned from instrument */
   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   /* Scan(rdBuf, "%d", setting); */

   /* Get if string returned from instrument */
   Scan(rdBuf, "%s", resBuf0); 
   if((auMT8820_status = AuMT8820_StringToEnum(commandArray, resBuf0, setting)) < 0) 
     return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set RF Level Output Mode
       Command(s): LVL[?]
       Purpose:    Sets/Gets the RF level output to On or Off.
       Setting defined by:  AUMT8820_RF_LEVEL_OUTPUT_MODE_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_RF_Level_Output_Mode(ViSession instrSession, ViInt32 setting)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"ON", "OFF", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   if (AuMT8820_InvalidViInt32Range(setting, 0, 1) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   len = sprintf(buf, "LVL %s", commandArray[setting]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get RF Level Output Mode
       Command(s): LVL[?]
       Purpose:    Sets/Gets the RF level output to On or Off.
       Setting defined by:  AUMT8820_RF_LEVEL_OUTPUT_MODE_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_RF_Level_Output_Mode(ViSession instrSession, ViInt32  _VI_FAR *setting)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"ON", "OFF", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViChar    resBuf0[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "LVL?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   /* Get if numeric returned from instrument */
   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   /* Scan(rdBuf, "%d", setting); */

   /* Get if string returned from instrument */
   Scan(rdBuf, "%s", resBuf0); 
   if((auMT8820_status = AuMT8820_StringToEnum(commandArray, resBuf0, setting)) < 0) 
     return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Level Continuous
       Command(s): LVLCONT[?]
       Purpose:    Sets the un-intertuptable change for the output level to On or Off. When set to On, the output level can be changed without signal interuption from thr level at setting to 30 dB lower level.
       Setting defined by:  AUMT8820_LEVEL_CONTINUOUS_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Level_Continuous(ViSession instrSession, ViInt32 setting)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"ON", "OFF", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   if (AuMT8820_InvalidViInt32Range(setting, 0, 1) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   len = sprintf(buf, "LVLCONT %s", commandArray[setting]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Level Continuous
       Command(s): LVLCONT[?]
       Purpose:    Sets the un-intertuptable change for the output level to On or Off. When set to On, the output level can be changed without signal interuption from thr level at setting to 30 dB lower level.
       Setting defined by:  AUMT8820_LEVEL_CONTINUOUS_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Level_Continuous(ViSession instrSession, ViInt32  _VI_FAR *setting)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"ON", "OFF", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViChar    resBuf0[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "LVLCONT?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   /* Get if numeric returned from instrument */
   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   /* Scan(rdBuf, "%d", setting); */

   /* Get if string returned from instrument */
   Scan(rdBuf, "%s", resBuf0); 
   if((auMT8820_status = AuMT8820_StringToEnum(commandArray, resBuf0, setting)) < 0) 
     return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Modulation Analysis Storage Mode
       Command(s): MA_STORAGE[?]
       Purpose:    Sets the storage mode for the waveform displayed in the modulation anaysis window.
       Mode defined by:  AUMT8820_MODULATION_ANALYSIS_STORAGE_MODE_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Modulation_Analysis_Storage_Mode(ViSession instrSession, ViInt32 mode)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"MAX", "MIN", "LATEST", "AVG", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   if (AuMT8820_InvalidViInt32Range(mode, 0, 3) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   len = sprintf(buf, "MA_STORAGE %s", commandArray[mode]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Modulation Analysis Storage Mode
       Command(s): MA_STORAGE[?]
       Purpose:    Sets the storage mode for the waveform displayed in the modulation anaysis window.
       Mode defined by:  AUMT8820_MODULATION_ANALYSIS_STORAGE_MODE_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Modulation_Analysis_Storage_Mode(ViSession instrSession, ViInt32  _VI_FAR *mode)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"MAX", "MIN", "LATEST", "AVG", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViChar    resBuf0[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "MA_STORAGE?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   /* Get if numeric returned from instrument */
   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   /* Scan(rdBuf, "%d", mode); */

   /* Get if string returned from instrument */
   Scan(rdBuf, "%s", resBuf0); 
   if((auMT8820_status = AuMT8820_StringToEnum(commandArray, resBuf0, mode)) < 0) 
     return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Magnitude Error
       Command(s): MAGTDERR?
       Purpose:    Gets the result for the last magnitude error measurement.
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Magnitude_Error_WCDMA(ViSession instrSession, ViReal64 _VI_FAR *percent)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "MAGTDERR?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%f", percent);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Carrier Frequency Error Maximum
       Command(s): MAX_CARRFERR?
       Purpose:    Gets the maximium value among the average count of frequency error measurement..
       Unit defined by:  AUMT8820_CARRIER_FREQ_ERROR_UNIT_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Carrier_Frequency_Error_Maximum_WCDMA(ViSession instrSession, ViInt32 unit, ViReal64  _VI_FAR *result)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"HZ", "PPM", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViChar    resBuf0[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   if (AuMT8820_InvalidViInt32Range(unit, 0, 1) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   len = sprintf(buf, "MAX_CARRFERR? %s", commandArray[unit]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   /* Get if numeric returned from instrument */
   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%f", result);

   /* Get if string returned from instrument */
   /*  Scan(rdBuf, "%s", resBuf0); */
   /*  if((auMT8820_status = AuMT8820_StringToEnum(commandArray, resBuf0, unit)) < 0) */
   /*    return auMT8820_status;*/

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Error Vector Magnitude Maximum
       Command(s): MAX_EVM?
       Purpose:    Queries the maximum value among average count of EVM measurements.
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Error_Vector_Magnitude_Maximum(ViSession instrSession, ViReal64 _VI_FAR *percent)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "MAX_EVM?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%f", percent);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Filtered Power Maximum
       Command(s): MAX_FILTPWR?
       Purpose:    Queries the maximum value among average count of RRC filtered transmission power measurements.
       Unit defined by:  AUMT8820_FILTERED_POWER_UNIT_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Filtered_Power_Maximum(ViSession instrSession, ViInt32 unit, ViReal64 _VI_FAR *power)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"DBM", "WATT", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   if (AuMT8820_InvalidViInt32Range(unit, 0, 1) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   len = sprintf(buf, "MAX_FILTPWR? %s", commandArray[unit]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%f", power);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get IQ Imbalance Maximum
       Command(s): MAX_IQIMB?
       Purpose:    Queries the maximum value among average count of IQ imbalance measurements.
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_IQ_Imbalance_Maximum(ViSession instrSession, ViReal64 _VI_FAR *percent)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "MAX_IQIMB?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%f", percent);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Magnitude Error Maximum
       Command(s): MAX_MAGTDERR?
       Purpose:    Queries the maximum value of magnitude error measurement results.
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Magnitude_Error_Maximum_WCDMA(ViSession instrSession, ViReal64 _VI_FAR *percent)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "MAX_MAGTDERR?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%f", percent);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Adjacent Channel Power Maximum
       Command(s): MAX_MODPWR[?]
       Purpose:    Queries the maximum value among average count of adjacent channel leakage power measurements.
       Offset defined by:  AUMT8820_ADJACENT_CHAN_POWER_OFFSET_xxx
       Unit defined by:  AUMT8820_ADJACENT_CHAN_POWER_UNIT_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Adjacent_Channel_Power_Maximum(ViSession instrSession, ViInt32 offset, ViInt32 unit, ViReal64  _VI_FAR *level)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"LOW10", "LOW5", "UP5", "UP10", 0L};
   static ViString commandArray2[] = {"DB", "DBM", "WATT", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViChar    resBuf0[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   if (AuMT8820_InvalidViInt32Range(offset, 0, 3) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   if (AuMT8820_InvalidViInt32Range(unit, 0, 2) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   len = sprintf(buf, "MAX_MODPWR? %s,%s", commandArray[offset], commandArray2[unit]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   /* Get if numeric returned from instrument */
   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%f", level);

   /* Get if string returned from instrument */
   /*  Scan(rdBuf, "%s", resBuf0); */
   /*  if((auMT8820_status = AuMT8820_StringToEnum(commandArray, resBuf0, offset)) < 0) */
   /*    return auMT8820_status;*/

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Origin Offset Maximum
       Command(s): MAX_ORGNOFS?
       Purpose:    Queries the maximum value among average count of origin offset measurements.
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Origin_Offset_Maximum(ViSession instrSession, ViReal64 _VI_FAR *level_dB)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "MAX_ORGNOFS?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%f", level_dB);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Peak Code Domain Error Maximum
       Command(s): MAX_PCDERR?
       Purpose:    Queries the maximum value among average count of peak code domain error measurements.
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Peak_Code_Domain_Error_Maximum(ViSession instrSession, ViReal64 _VI_FAR *level_dB)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "MAX_PCDERR?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%f", level_dB);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get RMS Phase Vector Error Maximum
       Command(s): MAX_PHASEERR?
       Purpose:    Gets the maximum value among measured results for average count of phase vector error.
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_RMS_Phase_Vector_Error_Maximum(ViSession instrSession, ViReal64 _VI_FAR *degrees)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "MAX_PHASEERR?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%f", degrees);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get TX Power Maximum WCDMA
       Command(s): MAX_POWER?
       Purpose:    Queries the maximum value among average count of TX power measurements.
       Unit defined by:  AUMT8820_TX_POWER_UNIT_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_TX_Power_Maximum_WCDMA(ViSession instrSession, ViInt32 unit, ViReal64 _VI_FAR *level)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"DBM", "WATT", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   if (AuMT8820_InvalidViInt32Range(unit, 0, 1) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   len = sprintf(buf, "MAX_POWER? %s", commandArray[unit]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%f", level);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Peak Vector Error Maximum
       Command(s): MAX_PVECTERR?
       Purpose:    Queries the maximum value among average count of peak vector error measurements.
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Peak_Vector_Error_Maximum(ViSession instrSession, ViReal64 _VI_FAR *percent)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "MAX_PVECTERR?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%f", percent);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Timing Error Maximum
       Command(s): MAX_TMGERR?
       Purpose:    Queries the maximum value of timing error measurement results for average count.
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Timing_Error_Maximum(ViSession instrSession, ViReal64 _VI_FAR *chip)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "MAX_TMGERR?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%f", chip);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Downlink RAB Rate Maximum
       Command(s): MAXRATE[?]
       Purpose:    Get/Set the maximum downlink rate of prioritzed RAB's
       Rate defined by:  AUMT8820_DOWNLINK_RAB_RATE_MAXIMUM_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Downlink_RAB_Rate_Maximum(ViSession instrSession, ViInt32 rate)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"12.2", "64", "144", "384", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   if (AuMT8820_InvalidViInt32Range(rate, 0, 3) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   len = sprintf(buf, "MAXRATE %s", commandArray[rate]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Downlink RAB Rate Maximum
       Command(s): MAXRATE[?]
       Purpose:    Get/Set the maximum downlink rate of prioritzed RAB's
       Rate defined by:  AUMT8820_DOWNLINK_RAB_RATE_MAXIMUM_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Downlink_RAB_Rate_Maximum(ViSession instrSession, ViInt32  _VI_FAR *rate)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"12.2", "64", "144", "384", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViChar    resBuf0[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "MAXRATE?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   /* Get if numeric returned from instrument */
   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   /* Scan(rdBuf, "%d", rate); */

   /* Get if string returned from instrument */
   Scan(rdBuf, "%s", resBuf0); 
   if((auMT8820_status = AuMT8820_StringToEnum(commandArray, resBuf0, rate)) < 0) 
     return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Location ID MCC
       Command(s): MCC[?]
       Purpose:    Sets/Gets the MCC for location area identification.
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Location_ID_MCC_WCDMA(ViSession instrSession, ViInt32 iD)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   if (AuMT8820_InvalidViInt32Range(iD,0,999) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   len = sprintf(buf, "MCC %d", iD);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Location ID MCC
       Command(s): MCC[?]
       Purpose:    Sets/Gets the MCC for location area identification.
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Location_ID_MCC_WCDMA(ViSession instrSession,  ViInt32 _VI_FAR *iD)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "MCC?");
   
  if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%d", iD);

   return auMT8820_status;

}

/*****************************************************************************/
/*
       Function:   Get Software Version
       Command(s): MCFV?
       Purpose:    Query the software version used on the MT8820A
       Mode defined by:  AUMT8820_SOFTWARE_OBJECT_xxx
 */
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Software_Version_WCDMA(ViSession instrSession, ViChar _VI_FAR *version)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
      
   len = sprintf(buf, "MCFV?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;

   Scan(rdBuf, "%c%c%c%c%c%c%c%c%c%c%c%c%c%c%c%c%c%c%c%c%c%c",version,(version+1),(version+2),(version+3),(version+4),(version+5),(version+6),(version+7),(version+8),(version+9),(version+10),(version+11),(version+12),(version+13),(version+14),(version+15),(version+16),(version+17),(version+18),(version+19),(version+20),(version+21));
   
   

   return auMT8820_status;

}




/*****************************************************************************/
/*
       Function:   Set Fundamental Measurement Mode
       Command(s): MEASMODE[?]
       Purpose:    Set/Get the fundamental measurement mode
       Mode defined by:  AUMT8820_FUNDAMENTAL_MEASUREMENT_MODE_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Fundamental_Measurement_Mode_WCDMA(ViSession instrSession, ViInt32 mode)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"NORM", "FAST", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   if (AuMT8820_InvalidViInt32Range(mode, 0, 1) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   len = sprintf(buf, "MEASMODE %s", commandArray[mode]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Fundamental Measurement Mode
       Command(s): MEASMODE[?]
       Purpose:    Set/Get the fundamental measurement mode
       Mode defined by:  AUMT8820_FUNDAMENTAL_MEASUREMENT_MODE_xxx
       Defined in MT8820 GSM LabWindows Driver.c
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Fundamental_Measurement_Mode_WCDMA(ViSession instrSession, ViInt32  _VI_FAR *mode)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"NORM", "FAST", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViChar    resBuf0[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "MEASMODE?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   /* Get if numeric returned from instrument */
   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   /* Scan(rdBuf, "%d", mode); */

   /* Get if string returned from instrument */
   Scan(rdBuf, "%s", resBuf0); 
   if((auMT8820_status = AuMT8820_StringToEnum(commandArray, resBuf0, mode)) < 0) 
     return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Measurement Object WCDMA
       Command(s): MEASOBJ[?]
       Purpose:    Sets/Gets the measurement object for time domain measurements.
       Object defined by:  AUMT8820_MEASUREMENT_OBJECT_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Measurement_Object_WCDMA(ViSession instrSession, ViInt32 object)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"RACHTMSK","RACH", "ILPC", "OTHER", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   if (AuMT8820_InvalidViInt32Range(object, 0, 3) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   len = sprintf(buf, "MEASOBJ %s", commandArray[object]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Measurement Object WCDMA
       Command(s): MEASOBJ[?]
       Purpose:    Sets/Gets the measurement object for time domain measurements.
       Object defined by:  AUMT8820_MEASUREMENT_OBJECT_WCDMA_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Measurement_Object_WCDMA(ViSession instrSession, ViInt32  _VI_FAR *object)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"RACHTMSK","RACH", "ILPC", "OTHER", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViChar    resBuf0[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "MEASOBJ?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   /* Get if numeric returned from instrument */
   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   /* Scan(rdBuf, "%d", object); */

   /* Get if string returned from instrument */
   Scan(rdBuf, "%s", resBuf0); 
   if((auMT8820_status = AuMT8820_StringToEnum(commandArray, resBuf0, object)) < 0) 
     return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Stop Measurement
       Command(s): MEASSTOP
       Purpose:    Stops the current measurement
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Stop_Measurement_WCDMA(ViSession instrSession)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   len = sprintf(buf, "MEASSTOP");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Carrier Frequency Error Minimum
       Command(s): MIN_CARRFERR[?]
       Purpose:    Gets the minimum value among the average count of frequency error measurement..
       Unit defined by:  AUMT8820_CARRIER_FREQ_ERROR_UNIT_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Carrier_Frequency_Error_Minimum_WCDMA(ViSession instrSession, ViInt32 unit, ViReal64  _VI_FAR *level)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"HZ", "PPM", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViChar    resBuf0[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   if (AuMT8820_InvalidViInt32Range(unit, 0, 1) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   len = sprintf(buf, "MIN_CARRFERR? %s", commandArray[unit]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   /* Get if numeric returned from instrument */
   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%f", level);

   /* Get if string returned from instrument */
   /*  Scan(rdBuf, "%s", resBuf0); */
   /*  if((auMT8820_status = AuMT8820_StringToEnum(commandArray, resBuf0, unit)) < 0) */
   /*    return auMT8820_status;*/

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Error Vector Magnitude Minimum
       Command(s): MIN_EVM?
       Purpose:    Queries the minimum value of average count measurement results for EVM.
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Error_Vector_Magnitude_Minimum(ViSession instrSession, ViReal64 _VI_FAR *percent)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "MIN_EVM?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%f", percent);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Filtered Power Minimum
       Command(s): MIN_FILTPWR?
       Purpose:    Queries the minimum value of average count results for RRC filtered transmission power.
       Unit defined by:  AUMT8820_FILTERED_POWER_UNIT_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Filtered_Power_Minimum(ViSession instrSession,  ViInt32 unit, ViReal64 _VI_FAR *power)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"DBM", "WATT", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   if (AuMT8820_InvalidViInt32Range(unit, 0, 1) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   len = sprintf(buf, "MIN_FILTPWR? %s", commandArray[unit]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%f", power);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get IQ Imbalance Minimum
       Command(s): MIN_IQIMB?
       Purpose:    Queries the minimum value of average count results for IQ imbalanced measurements.
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_IQ_Imbalance_Minimum(ViSession instrSession, ViReal64 _VI_FAR *result)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "MIN_IQIMB?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%f", result);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Magnitude Error Minimum
       Command(s): MIN_MAGTDERR?
       Purpose:    Queries the minimum value of magnitude error measurement results.
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Magnitude_Error_Minimum_WCDMA(ViSession instrSession, ViReal64 _VI_FAR *percent)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "MIN_MAGTDERR?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%f", percent);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Adjacent Channel Power Minimum
       Command(s): MIN_MODPWR?
       Purpose:    Queries the minimum value of average count results for adjacent channel power measurements.
       Offset defined by:  AUMT8820_ADJACENT_CHAN_POWER_OFFSET_xxx
       Unit defined by:  AUMT8820_ADJACENT_CHAN_POWER_UNIT_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Adjacent_Channel_Power_Minimum(ViSession instrSession, ViInt32 offset, ViInt32 unit, ViReal64 _VI_FAR *level)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"LOW10", "LOW5", "UP5", "UP10", 0L};
   static ViString commandArray2[] = {"DB", "DBM", "WATT", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   if (AuMT8820_InvalidViInt32Range(offset, 0, 3) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   if (AuMT8820_InvalidViInt32Range(unit, 0, 2) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   len = sprintf(buf, "MIN_MODPWR? %s,%s", commandArray[offset], commandArray2[unit]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%f", level);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Origin Offset Minimum
       Command(s): MIN_ORGNOFS?
       Purpose:    Queries the minimum value of average count results for origin offset measurements.
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Origin_Offset_Minimum(ViSession instrSession, ViReal64 _VI_FAR *level_dB)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "MIN_ORGNOFS?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%f", level_dB);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Peak Code Domain Error Minimum
       Command(s): MIN_PCDERR?
       Purpose:    Queries the minimum value of average count results for peak code domain measurements.
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Peak_Code_Domain_Error_Minimum(ViSession instrSession, ViReal64 _VI_FAR *level_dB)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "MIN_PCDERR?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%f", level_dB);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get RMS Phase Vector Error Minimum
       Command(s): MIN_PHASEERR?
       Purpose:    Gets the mimimum value among measured results for average count of phase vector error.
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_RMS_Phase_Vector_Error_Minimum(ViSession instrSession, ViReal64 _VI_FAR *degree)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "MIN_PHASEERR?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%f", degree);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get TX Power Minimum
       Command(s): MIN_POWER?
       Purpose:    Queries the minimum value of average count results for TX power measurements.
       Unit defined by:  AUMT8820_TX_POWER_UNIT_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_TX_Power_Minimum_WCDMA(ViSession instrSession, ViInt32 unit, ViReal64 _VI_FAR *level)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"DBM", "WATT", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   if (AuMT8820_InvalidViInt32Range(unit, 0, 1) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   len = sprintf(buf, "MIN_POWER? %s", commandArray[unit]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%f", level);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Peak Vector Error Minimum
       Command(s): MIN_PVECTERR?
       Purpose:    Queries the minimum value of average count results for peak vector error measurements.
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Peak_Vector_Error_Minimum(ViSession instrSession, ViReal64 _VI_FAR *percent)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "MIN_PVECTERR?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%f", percent);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Timing Error Minimum
       Command(s): MIN_TMGERR?
       Purpose:    Queries the minimum value of average count results for timing error measurements.
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Timing_Error_Minimum(ViSession instrSession, ViReal64 _VI_FAR *chip)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "MIN_TMGERR?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%f", chip);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Marker 1 Mode
       Command(s): MKR1_TDM[?]
       Purpose:    Sets the marker mode for marker 1 to On or Off.
       Setting defined by:  AUMT8820_MARKER_1_MODE_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Marker_1_Mode(ViSession instrSession, ViInt32 setting)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"ON", "OFF", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   if (AuMT8820_InvalidViInt32Range(setting, 0, 1) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   len = sprintf(buf, "MKR1_TDM %s", commandArray[setting]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Marker 1 Mode
       Command(s): MKR1_TDM[?]
       Purpose:    Sets the marker mode for marker 1 to On or Off.
       Setting defined by:  AUMT8820_MARKER_1_MODE_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Marker_1_Mode(ViSession instrSession, ViInt32  _VI_FAR *setting)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"ON", "OFF", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViChar    resBuf0[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "MKR1_TDM?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   /* Get if numeric returned from instrument */
   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   /* Scan(rdBuf, "%d", setting); */

   /* Get if string returned from instrument */
   Scan(rdBuf, "%s", resBuf0); 
   if((auMT8820_status = AuMT8820_StringToEnum(commandArray, resBuf0, setting)) < 0) 
     return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Marker 2 Mode
       Command(s): MKR2_TDM[?]
       Purpose:    Sets the marker mode for marker 2 to On or Off.
       Setting defined by:  AUMT8820_MARKER_2_MODE_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Marker_2_Mode(ViSession instrSession, ViInt32 setting)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"ON", "OFF", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   if (AuMT8820_InvalidViInt32Range(setting, 0, 1) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   len = sprintf(buf, "MKR2_TDM %s", commandArray[setting]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Marker 2 Mode
       Command(s): MKR2_TDM[?]
       Purpose:    Sets the marker mode for marker 2 to On or Off.
       Setting defined by:  AUMT8820_MARKER_2_MODE_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Marker_2_Mode(ViSession instrSession, ViInt32  _VI_FAR *setting)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"ON", "OFF", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViChar    resBuf0[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "MKR2_TDM?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   /* Get if numeric returned from instrument */
   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   /* Scan(rdBuf, "%d", setting); */

   /* Get if string returned from instrument */
   Scan(rdBuf, "%s", resBuf0); 
   if((auMT8820_status = AuMT8820_StringToEnum(commandArray, resBuf0, setting)) < 0) 
     return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Spectrum Emission Mask Marker Frequency
       Command(s): MKRF_SMASK[?]
       Purpose:    Sets the marker position in the spectrum emission mask view window.
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Spectrum_Emission_Mask_Marker_Frequency(ViSession instrSession, ViReal64 frequency_MHz)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   len = sprintf(buf, "MKRF_SMASK %.2fMHZ", frequency_MHz);
   if (AuMT8820_InvalidViReal64Range(frequency_MHz, -12.510000, 12.510000) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Spectrum Emission Mask Marker Frequency
       Command(s): MKRF_SMASK[?]
       Purpose:    Sets the marker position in the spectrum emission mask view window.
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Spectrum_Emission_Mask_Marker_Frequency(ViSession instrSession, ViReal64 _VI_FAR *frequency)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "MKRF_SMASK?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%f", frequency);

   return auMT8820_status;

}
 

/*****************************************************************************/
/*
       Function:   Set Modulation Analysis Marker
       Command(s): MKR_MOD[?]
       Purpose:    Sets the marker display in the modulation analysis window to On or Off.
       Mode defined by:  AUMT8820_MODULATION_ANALYSIS_MARKER_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Modulation_Analysis_Marker(ViSession instrSession, ViInt32 mode)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"NRM", "OFF", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   if (AuMT8820_InvalidViInt32Range(mode, 0, 1) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   len = sprintf(buf, "MKR_MOD %s", commandArray[mode]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Modulation Analysis Marker
       Command(s): MKR_MOD[?]
       Purpose:    Sets the marker display in the modulation analysis window to On or Off.
       Mode defined by:  AUMT8820_MODULATION_ANALYSIS_MARKER_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Modulation_Analysis_Marker(ViSession instrSession, ViInt32  _VI_FAR *mode)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"NRM", "OFF", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViChar    resBuf0[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "MKR_MOD?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   /* Get if numeric returned from instrument */
   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   /* Scan(rdBuf, "%d", mode); */

   /* Get if string returned from instrument */
   Scan(rdBuf, "%s", resBuf0); 
   if((auMT8820_status = AuMT8820_StringToEnum(commandArray, resBuf0, mode)) < 0) 
     return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Spectrum Emission Mask Marker Mode
       Command(s): MKR_SMASK[?]
       Purpose:    Sets the marker display in the spectrum emission mask window to On or Off.
       Mode defined by:  AUMT8820_SPECTRUM_EMISSION_MASK_MARKER_MODE_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Spectrum_Emission_Mask_Marker_Mode(ViSession instrSession, ViInt32 mode)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"NRM", "OFF", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   if (AuMT8820_InvalidViInt32Range(mode, 0, 1) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   len = sprintf(buf, "MKR_SMASK %s", commandArray[mode]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Spectrum Emission Mask Marker Mode
       Command(s): MKR_SMASK[?]
       Purpose:    Sets the marker display in the spectrum emission mask window to On or Off.
       Mode defined by:  AUMT8820_SPECTRUM_EMISSION_MASK_MARKER_MODE_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Spectrum_Emission_Mask_Marker_Mode(ViSession instrSession, ViInt32  _VI_FAR *mode)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"NRM", "OFF", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViChar    resBuf0[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "MKR_SMASK?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   /* Get if numeric returned from instrument */
   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   /* Scan(rdBuf, "%d", mode); */

   /* Get if string returned from instrument */
   Scan(rdBuf, "%s", resBuf0); 
   if((auMT8820_status = AuMT8820_StringToEnum(commandArray, resBuf0, mode)) < 0) 
     return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Marker Position
       Command(s): MKR_TDM[?]
       Purpose:    Sets the marker postion in the result window for time domain measurement.
       Marker defined by:  AUMT8820_MARKER_POSITION_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Marker_Position(ViSession instrSession, ViInt32 marker, ViReal64 position)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"", "MKR1", "MKR2", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   if (AuMT8820_InvalidViReal64Range(position, -15.000000, 1515.00000) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   if (AuMT8820_InvalidViInt32Range(marker, 0, 2) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   len = sprintf(buf, "MKR_TDM %.4f,%s", position, commandArray[marker]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Marker Position
       Command(s): MKR_TDM[?]
       Purpose:    Sets the marker postion in the result window for time domain measurement.
       Marker defined by:  AUMT8820_MARKER_POSITION_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Marker_Position(ViSession instrSession, ViInt32 marker, ViReal64 *position)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {" ","MKR1", "MKR2", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViChar    resBuf0[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "MKR_TDM? %s", commandArray[marker]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   /* Get if numeric returned from instrument */
   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%f", position);

   /* Get if string returned from instrument */
   /*  Scan(rdBuf, "%s", resBuf0); */
   /*  if((auMT8820_status = AuMT8820_StringToEnum(commandArray, resBuf0, marker)) < 0) */
   /*    return auMT8820_status;*/

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Marker Relative Level
       Command(s): MKRLREL_TDM?
       Purpose:    Queries the relative level difference between marker 1 and 2 in the time domain window.
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Marker_Relative_Level(ViSession instrSession, ViReal64 _VI_FAR *level_dB)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "MKRLREL_TDM?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%f", level_dB);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Modulation Analysis Marker Level
       Command(s): MKRL_MOD?
       Purpose:    Queries the marker level in the modulation analysis window.
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Modulation_Analysis_Marker_Level(ViSession instrSession, ViReal64 _VI_FAR *level)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "MKRL_MOD?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%f", level);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Spectrum Emission Mask Marker Level
       Command(s): MKRL_SMASK?
       Purpose:    Queries the marker position in the spectrum emission mask view window.
       Unit defined by:  AUMT8820_SPECTRUM_EMISSION_MARKER_UNIT_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Spectrum_Emission_Mask_Marker_Level(ViSession instrSession, ViInt32 unit, ViReal64 _VI_FAR *level)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"DB", "DBM", "WATT", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   if (AuMT8820_InvalidViInt32Range(unit, 0, 2) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   len = sprintf(buf, "MKRL_SMASK? %s", commandArray[unit]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%f", level);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Spectrum Monitor Normal Marker Level
       Command(s): MKRL_SPM?
       Purpose:    Queries the marker point level of the normal marker in the spectrum monitor screen.
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Spectrum_Monitor_Normal_Marker_Level(ViSession instrSession, ViInt32 number, ViReal64 _VI_FAR *level_dBm)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   if (AuMT8820_InvalidViInt32Range(number, 1, 5) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   len = sprintf(buf, "MKRL_SPM? %d", number);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%f", level_dBm);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Time Domain Marker Level
       Command(s): MKRL_TDM?
       Purpose:    Queries the power at the marker position in the time domain measurement result window.
       Marker defined by:  AUMT8820_MARKER_POSITION_MARKER_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Time_Domain_Marker_Level(ViSession instrSession, ViInt32 marker, ViReal64 _VI_FAR *level_dBm)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"MKR1", "MKR2", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   if (AuMT8820_InvalidViInt32Range(marker, 0, 1) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   len = sprintf(buf, "MKRL_TDM? %s", commandArray[marker]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%f", level_dBm);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Modulation Analysis Marker Position
       Command(s): MKRP_MOD[?]
       Purpose:    Sets/Gets the marker position in the modulation analysis view window.
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Modulation_Analysis_Marker_Position(ViSession instrSession, ViInt32 position)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   if (AuMT8820_InvalidViInt32Range(position,0,2559) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   len = sprintf(buf, "MKRP_MOD %d", position);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Modulation Analysis Marker Position
       Command(s): MKRP_MOD[?]
       Purpose:    Sets/Gets the marker position in the modulation analysis view window.
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Modulation_Analysis_Marker_Position(ViSession instrSession,  ViInt32 _VI_FAR *position)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "MKRP_MOD?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%d", position);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Spectrum Monitor Marker Position
       Command(s): MKRP_SPM[?]
       Purpose:    Sets/Gets the marker position in the spectrum monitor screen.
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Spectrum_Monitor_Marker_Position(ViSession instrSession, ViChar _VI_FAR frequency[])
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   /*if (AuMT8820_InvalidViInt32Range(frequency,1,5) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE; */
   len = sprintf(buf, "MKRP_SPM %s", frequency);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Spectrum Monitor Marker Position
       Command(s): MKRP_SPM[?]
       Purpose:    Sets/Gets the marker position in the spectrum monitor screen.
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Spectrum_Monitor_Marker_Position(ViSession instrSession,  ViChar _VI_FAR frequency[])
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "MKRP_SPM?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%s", frequency);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Marker Relative Slot
       Command(s): MKRSREL_TDM?
       Purpose:    Queries the relative slot difference between marker 1 and 2 in the time domain measurement result window.
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Marker_Relative_Slot(ViSession instrSession, ViInt32 _VI_FAR *slot)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "MKRSREL_TDM?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%d", slot);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Marker Relative Time
       Command(s): MKRTREL_TDM?
       Purpose:    Queries the relative time difference between marker 1 and 2 in the time domain measurement result window.
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Marker_Relative_Time(ViSession instrSession, ViReal64 _VI_FAR *time_ms)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "MKRTREL_TDM?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%f", time_ms);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Marker Type
       Command(s): MKRTYPE[?]
       Purpose:    Sets the marker typein the time domain measurement window.
       Marker_Type defined by:  AUMT8820_MARKER_TYPE_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Marker_Type(ViSession instrSession, ViInt32 marker_Type)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"NRM", "SLOT", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   if (AuMT8820_InvalidViInt32Range(marker_Type, 0, 1) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   len = sprintf(buf, "MKRTYPE %s", commandArray[marker_Type]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Marker Type
       Command(s): MKRTYPE[?]
       Purpose:    Sets the marker typein the time domain measurement window.
       Marker_Type defined by:  AUMT8820_MARKER_TYPE_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Marker_Type(ViSession instrSession, ViInt32  _VI_FAR *marker_Type)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"NRM", "SLOT", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViChar    resBuf0[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "MKRTYPE?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   /* Get if numeric returned from instrument */
   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   /* Scan(rdBuf, "%d", marker_Type); */

   /* Get if string returned from instrument */
   Scan(rdBuf, "%s", resBuf0); 
   if((auMT8820_status = AuMT8820_StringToEnum(commandArray, resBuf0, marker_Type)) < 0) 
     return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Mmax Value For RACH Parameter
       Command(s): MMAX[?]
       Purpose:    Sets/Gets the Mmax value for RACH parameter.
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Mmax_Value_For_RACH_Parameter(ViSession instrSession, ViInt32 number)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   if (AuMT8820_InvalidViInt32Range(number,1,32) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   len = sprintf(buf, "MMAX %d", number);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Mmax Value For RACH Parameter
       Command(s): MMAX[?]
       Purpose:    Sets/Gets the Mmax value for RACH parameter.
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Mmax_Value_For_RACH_Parameter(ViSession instrSession,  ViInt32 *number)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "MMAX?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%d", number);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Location ID MNC
       Command(s): MNC[?]
       Purpose:    Sets/Gets the MNC for location area identification.
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Location_ID_MNC_WCDMA(ViSession instrSession, ViInt32 iD)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   if (AuMT8820_InvalidViInt32Range(iD,0,999) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   len = sprintf(buf, "MNC %02d", iD);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Location ID MNC
       Command(s): MNC[?]
       Purpose:    Sets/Gets the MNC for location area identification.
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Location_ID_MNC_WCDMA(ViSession instrSession,  ViInt32 _VI_FAR *iD)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "MNC?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%d", iD);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Adjacent Channel Power
       Command(s): MODPWR?
       Purpose:    Get the adjacent channel lpower level
       Offset defined by:  AUMT8820_ADJACENT_CHAN_POWER_OFFSET_xxx
       Unit defined by:  AUMT8820_ADJACENT_CHAN_POWER_UNIT_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Adjacent_Channel_Power(ViSession instrSession, ViInt32 offset, ViInt32 unit, ViReal64 _VI_FAR *level)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"LOW10", "LOW5", "UP5", "UP10", 0L};
   static ViString commandArray2[] = {"DB", "DBM", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   if (AuMT8820_InvalidViInt32Range(offset, 0, 3) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   if (AuMT8820_InvalidViInt32Range(unit, 0, 1) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   len = sprintf(buf, "MODPWR? %s,%s", commandArray[offset], commandArray2[unit]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%f", level);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Average Count for Modulation Analysis Measurement
       Command(s): MOD_AVG[?]
       Purpose:    Set/Get the average count for the modulation analysis measurement
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Average_Count_for_Modulation_Analysis_Measurement(ViSession instrSession, ViInt32 count)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   if (AuMT8820_InvalidViInt32Range(count,1,9999) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   len = sprintf(buf, "MOD_AVG %d", count);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Average Count for Modulation Analysis Measurement
       Command(s): MOD_AVG[?]
       Purpose:    Set/Get the average count for the modulation analysis measurement
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Average_Count_for_Modulation_Analysis_Measurement(ViSession instrSession,  ViInt32 _VI_FAR *count)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "MOD_AVG?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%d", count);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Modulation Analysis Measurement Mode
       Command(s): MOD_MEAS[?]
       Purpose:    Set/Get the modulation analysis measurement mode
       Mode defined by:  AUMT8820_MODULATION_ANALYSIS_MEASUREMENT_MODE_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Modulation_Analysis_Measurement_Mode(ViSession instrSession, ViInt32 mode)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"ON", "OFF", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   if (AuMT8820_InvalidViInt32Range(mode, 0, 1) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   len = sprintf(buf, "MOD_MEAS %s", commandArray[mode]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Modulation Analysis Measurement Mode
       Command(s): MOD_MEAS[?]
       Purpose:    Set/Get the modulation analysis measurement mode
       Mode defined by:  AUMT8820_MODULATION_ANALYSIS_MEASUREMENT_MODE_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Modulation_Analysis_Measurement_Mode(ViSession instrSession, ViInt32  _VI_FAR *mode)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"ON", "OFF", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViChar    resBuf0[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "MOD_MEAS?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   /* Get if numeric returned from instrument */
   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   /* Scan(rdBuf, "%d", mode); */

   /* Get if string returned from instrument */
   Scan(rdBuf, "%s", resBuf0); 
   if((auMT8820_status = AuMT8820_StringToEnum(commandArray, resBuf0, mode)) < 0) 
     return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Modulation Analysis Waveform
       Command(s): MOD_WVFORM[?]
       Purpose:    Sets/Gets the waveform to be displayed in the modulaion analysis view window.
       Item defined by:  AUMT8820_MODULATION_ANALYSIS_WAVEFORM_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Modulation_Analysis_Waveform(ViSession instrSession, ViInt32 item)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"EVM", "PHASE", "MAGTD", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   if (AuMT8820_InvalidViInt32Range(item, 0, 2) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   len = sprintf(buf, "MOD_WVFORM %s", commandArray[item]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Modulation Analysis Waveform
       Command(s): MOD_WVFORM[?]
       Purpose:    Sets/Gets the waveform to be displayed in the modulaion analysis view window.
       Item defined by:  AUMT8820_MODULATION_ANALYSIS_WAVEFORM_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Modulation_Analysis_Waveform(ViSession instrSession, ViInt32  _VI_FAR *item)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"EVM", "PHASE", "MAGTD", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViChar    resBuf0[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "MOD_WVFORM?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   /* Get if numeric returned from instrument */
   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   /* Scan(rdBuf, "%d", item); */

   /* Get if string returned from instrument */
   Scan(rdBuf, "%s", resBuf0); 
   if((auMT8820_status = AuMT8820_StringToEnum(commandArray, resBuf0, item)) < 0) 
     return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Measurement Status
       Command(s): MSTAT?
       Purpose:    Returns measurement status
       Status defined by:  AUMT8820_MEASUREMENT_STATUS_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Measurement_Status_WCDMA(ViSession instrSession, ViInt32 _VI_FAR *status)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViChar    resBuf0[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "MSTAT?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   /* Get if numeric returned from instrument */
   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%d", status);

   /* Get if string returned from instrument */
   /*  Scan(rdBuf, "%s", resBuf0); */
   /*  if((auMT8820_status = AuMT8820_StringToEnum(commandArray, resBuf0, status)) < 0) */
   /*    return auMT8820_status;*/

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set NB01 Max
       Command(s): NB01MAX[?]
       Purpose:    Sets/Gets the NB01 max value for RACH.
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_NB01_Max(ViSession instrSession, ViInt32 number)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   if (AuMT8820_InvalidViInt32Range(number,0,50) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   len = sprintf(buf, "NB01MAX %d", number);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get NB01 Max
       Command(s): NB01MAX[?]
       Purpose:    Sets/Gets the NB01 max value for RACH.
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_NB01_Max(ViSession instrSession,  ViInt32 _VI_FAR *number)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "NB01MAX?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%d", number);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set NB01 Min
       Command(s): NB01MIN[?]
       Purpose:    Sets/Gets the NB01 min value for RACH.
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_NB01_Min(ViSession instrSession, ViInt32 number)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   if (AuMT8820_InvalidViInt32Range(number,0,50) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   len = sprintf(buf, "NB01MIN %d", number);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get NB01 Min
       Command(s): NB01MIN[?]
       Purpose:    Sets/Gets the NB01 min value for RACH.
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_NB01_Min(ViSession instrSession,  ViInt32 _VI_FAR *number)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "NB01MIN?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%d", number);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Noise Calibration for TDM
       Command(s): NOISECAL[?]
       Purpose:    Sets the noise calibration for time domain measurement.
       Switchmode defined by:  AUMT8820_NOISE_CALIBRATION_FOR_TDM_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Noise_Calibration_for_TDM(ViSession instrSession, ViInt32 switchmode)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"ON", "OFF", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   if (AuMT8820_InvalidViInt32Range(switchmode, 0, 1) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   len = sprintf(buf, "NOISECAL %s", commandArray[switchmode]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Noise Calibration for TDM
       Command(s): NOISECAL[?]
       Purpose:    Sets the noise calibration for time domain measurement.
       Switchmode defined by:  AUMT8820_NOISE_CALIBRATION_FOR_TDM_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Noise_Calibration_for_TDM(ViSession instrSession, ViInt32  _VI_FAR *switchmode)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"ON", "OFF", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViChar    resBuf0[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "NOISECAL?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   /* Get if numeric returned from instrument */
   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   /* Scan(rdBuf, "%d", switchmode); */

   /* Get if string returned from instrument */
   Scan(rdBuf, "%s", resBuf0); 
   if((auMT8820_status = AuMT8820_StringToEnum(commandArray, resBuf0, switchmode)) < 0) 
     return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Occupied Bandwidth
       Command(s): OBW?,OCCBW?
       Purpose:    Queries the measured results for occupied bandwidth.
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Occupied_Bandwidth(ViSession instrSession, ViReal64 _VI_FAR *frequency_Hz)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "OBW?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%f", frequency_Hz);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Occupied Bandwidth Frequency Mode
       Command(s): OBWFREQ?
       Purpose:    Queries the frequency that is 2 % of total power using upper and lower limits of measured waveform for occupied bandwidth.
       Offset defined by:  AUMT8820_OCCUPIED_BANDWIDTH_FREQUENCY_MODE_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Occupied_Bandwidth_Frequency_Mode(ViSession instrSession, ViInt32 offset, ViReal64 _VI_FAR *frequency_Hz)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"UPPER", "LOWER", "CENTER", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "OBWFREQ? %s", commandArray[offset]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%f", frequency_Hz);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Average Count for Occupied Bandwidth Measurement
       Command(s): OBW_AVG[?]
       Purpose:    Set/Get the average count for occupied bandwidth measurements
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Average_Count_for_Occupied_Bandwidth_Measurement(ViSession instrSession, ViInt32 count)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   if (AuMT8820_InvalidViInt32Range(count,1,9999) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   len = sprintf(buf, "OBW_AVG %d", count);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Average Count for Occupied Bandwidth Measurement
       Command(s): OBW_AVG[?]
       Purpose:    Set/Get the average count for occupied bandwidth measurements
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Average_Count_for_Occupied_Bandwidth_Measurement(ViSession instrSession,  ViInt32 _VI_FAR *count)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "OBW_AVG?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%d", count);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Occupied Bandwidth Detect Mode
       Command(s): OBW_DET[?]
       Purpose:    Sets the detection mode for occupied bandwidth.
       Mode defined by:  AUMT8820_OCCUPIED_BANDWIDTH_DETECT_MODE_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Occupied_Bandwidth_Detect_Mode(ViSession instrSession, ViInt32 mode)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"POS", "NEG", "SAMPLE", "AVG", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   if (AuMT8820_InvalidViInt32Range(mode, 0, 3) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   len = sprintf(buf, "OBW_DET %s", commandArray[mode]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Occupied Bandwidth Detect Mode
       Command(s): OBW_DET[?]
       Purpose:    Sets the detection mode for occupied bandwidth.
       Mode defined by:  AUMT8820_OCCUPIED_BANDWIDTH_DETECT_MODE_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Occupied_Bandwidth_Detect_Mode(ViSession instrSession, ViInt32  _VI_FAR *mode)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"POS", "NEG", "SAMPLE", "AVG", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViChar    resBuf0[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "OBW_DET?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   /* Get if numeric returned from instrument */
   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   /* Scan(rdBuf, "%d", mode); */

   /* Get if string returned from instrument */
   Scan(rdBuf, "%s", resBuf0); 
   if((auMT8820_status = AuMT8820_StringToEnum(commandArray, resBuf0, mode)) < 0) 
     return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Occupied Bandwidth Measurement Mode
       Command(s): OBW_MEAS[?]
       Purpose:    Set/Get the occupied bandwidth measurement mode
       Mode defined by:  AUMT8820_OCCUPIED_BANDWIDTH_MEASUREMENT_MODE_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Occupied_Bandwidth_Measurement_Mode(ViSession instrSession, ViInt32 mode)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"ON", "OFF", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   if (AuMT8820_InvalidViInt32Range(mode, 0, 1) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   len = sprintf(buf, "OBW_MEAS %s", commandArray[mode]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Occupied Bandwidth Measurement Mode
       Command(s): OBW_MEAS[?]
       Purpose:    Set/Get the occupied bandwidth measurement mode
       Mode defined by:  AUMT8820_OCCUPIED_BANDWIDTH_MEASUREMENT_MODE_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Occupied_Bandwidth_Measurement_Mode(ViSession instrSession, ViInt32  _VI_FAR *mode)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"ON", "OFF", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViChar    resBuf0[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "OBW_MEAS?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   /* Get if numeric returned from instrument */
   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   /* Scan(rdBuf, "%d", mode); */

   /* Get if string returned from instrument */
   Scan(rdBuf, "%s", resBuf0); 
   if((auMT8820_status = AuMT8820_StringToEnum(commandArray, resBuf0, mode)) < 0) 
     return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Occupied Bandwidth Ratio
       Command(s): OBW_RATIO[?]
       Purpose:    Sets the band occupation for occupied bandwidth measurement.
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Occupied_Bandwidth_Ratio(ViSession instrSession, ViReal64 ratio)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   len = sprintf(buf, "OBW_RATIO %.1f", ratio);
   if (AuMT8820_InvalidViReal64Range(ratio, 80.000000, 99.900000) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Occupied Bandwidth Ratio
       Command(s): OBW_RATIO[?]
       Purpose:    Sets the band occupation for occupied bandwidth measurement.
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Occupied_Bandwidth_Ratio(ViSession instrSession, ViReal64 _VI_FAR *ratio)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "OBW_RATIO?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%f", ratio);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Downlink OCNS
       Command(s): OCNS[?]
       Purpose:    Sets the OCNS downlink physical channel.
       Setting defined by:  AUMT8820_DOWNLINK_OCNS_xxx
       Not Implemented in MT8820
*/
/*****************************************************************************/
/*ViStatus _VI_FUNC AuMT8820_Set_Downlink_OCNS(ViSession instrSession, ViInt32 setting)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"ON", "OFF", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   if (AuMT8820_InvalidViInt32Range(setting, 0, 1) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   len = sprintf(buf, "OCNS %s", commandArray[setting]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}

*/
/*****************************************************************************/
/*
       Function:   Get Downlink OCNS
       Command(s): OCNS[?]
       Purpose:    Sets the OCNS downlink physical channel.
       Setting defined by:  AUMT8820_DOWNLINK_OCNS_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Downlink_OCNS(ViSession instrSession, ViInt32  _VI_FAR *setting)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"ON", "OFF", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViChar    resBuf0[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "OCNS?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   /* Get if numeric returned from instrument */
   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   /* Scan(rdBuf, "%d", setting); */

   /* Get if string returned from instrument */
   Scan(rdBuf, "%s", resBuf0); 
   if((auMT8820_status = AuMT8820_StringToEnum(commandArray, resBuf0, setting)) < 0) 
     return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Downlink OCNS Power
       Command(s): OCNSPWR?
       Purpose:    Queries the output level for downlink OCNS.
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Downlink_OCNS_Power(ViSession instrSession, ViReal64 _VI_FAR *level_dB)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "OCNSPWR?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%f", level_dB);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Output Level
       Command(s): OLVL[?],SGLVL
       Purpose:    Set/Get the signal generator output level
       Unit defined by:  AUMT8820_OUTPUT_LEVEL_UNIT_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Output_Level_WCDMA(ViSession instrSession, ViReal64 level, ViInt32 units)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString unitString[] = {"DB", "DBM", "DBU", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   if (AuMT8820_InvalidViReal64Range(level, -140.00, 113.00) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   if (AuMT8820_InvalidViInt32Range(units, 0, 2) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   len = sprintf(buf, "OLVL %f%s", level, unitString[units]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

 }


/*****************************************************************************/
/*
       Function:   Get Output Level
       Command(s): OLVL[?],SGLVL
       Purpose:    Set/Get the signal generator output level
       Unit defined by:  AUMT8820_OUTPUT_LEVEL_UNIT_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Output_Level_WCDMA(ViSession instrSession, ViReal64 _VI_FAR *level)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString unitString[] = {"DB", "DBM", "DBU", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViChar    resBuf0[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "OLVL?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%f%s", level, resBuf0);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Origin Offset
       Command(s): ORGNOFS?
       Purpose:    Get the origin offset in dB
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Origin_Offset(ViSession instrSession, ViReal64 _VI_FAR *level_dB)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "ORGNOFS?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%f", level_dB);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Path Loss
       Command(s): PATHLOSS?
       Purpose:    Queries the difference between CPICH power and CPICH RSCP.
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Path_Loss(ViSession instrSession, ViReal64 _VI_FAR *level_dB)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "PATHLOSS?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%f", level_dB);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get PCCH Signaling RB
       Command(s): PCCH?
       Purpose:    Queries the PCCH transmission rate for signaling RB's
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_PCCH_Signaling_RB(ViSession instrSession, ViReal64 _VI_FAR *rate)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "PCCH?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%f", rate);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Downlink PCCPCH
       Command(s): PCCPCH[?]
       Purpose:    Sets the PCCPCH output for downlink channel.
       Setting defined by:  AUMT8820_DOWNLINK_PCCPCH_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Downlink_PCCPCH(ViSession instrSession, ViInt32 setting)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"ON", "OFF", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   if (AuMT8820_InvalidViInt32Range(setting, 0, 1) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   len = sprintf(buf, "PCCPCH %s", commandArray[setting]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Downlink PCCPCH
       Command(s): PCCPCH[?]
       Purpose:    Sets the PCCPCH output for downlink channel.
       Setting defined by:  AUMT8820_DOWNLINK_PCCPCH_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Downlink_PCCPCH(ViSession instrSession, ViInt32  _VI_FAR *setting)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"ON", "OFF", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViChar    resBuf0[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "PCCPCH?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   /* Get if numeric returned from instrument */
   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   /* Scan(rdBuf, "%d", setting); */

   /* Get if string returned from instrument */
   Scan(rdBuf, "%s", resBuf0); 
   if((auMT8820_status = AuMT8820_StringToEnum(commandArray, resBuf0, setting)) < 0) 
     return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Downlink PCCPCH Power
       Command(s): PCCPCHPWR[?]
       Purpose:    Sets the output level for downlink PCCPCH.
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Downlink_PCCPCH_Power(ViSession instrSession, ViReal64 level_dB)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   len = sprintf(buf, "PCCPCHPWR %.1f", level_dB);
   if (AuMT8820_InvalidViReal64Range(level_dB, -30.000000, 0.000000) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Downlink PCCPCH Power
       Command(s): PCCPCHPWR[?]
       Purpose:    Sets the output level for downlink PCCPCH.
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Downlink_PCCPCH_Power(ViSession instrSession, ViReal64 _VI_FAR *level_dB)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "PCCPCHPWR?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%f", level_dB);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Peak Code Domain Error
       Command(s): PCDERR?
       Purpose:    Queries the peak code domain for the most recent measurement.
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Peak_Code_Domain_Error(ViSession instrSession, ViReal64 _VI_FAR *level_dB)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "PCDERR?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%f", level_dB);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Average Count for Peak Code Domain Error Measurement
       Command(s): PCDE_AVG[?]
       Purpose:    Set/Get the average count for the peak code domain error measurement
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Average_Count_for_Peak_Code_Domain_Error_Measurement(ViSession instrSession, ViInt32 count)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   if (AuMT8820_InvalidViInt32Range(count,1,9999) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   len = sprintf(buf, "PCDE_AVG %d", count);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Average Count for Peak Code Domain Error Measurement
       Command(s): PCDE_AVG[?]
       Purpose:    Set/Get the average count for the peak code domain error measurement
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Average_Count_for_Peak_Code_Domain_Error_Measurement(ViSession instrSession,  ViInt32 _VI_FAR *count)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "PCDE_AVG?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%d", count);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Peak Code Domain Error Measurement Mode
       Command(s): PCDE_MEAS[?]
       Purpose:    Set/Get the peak code domain error measurement mode
       Mode defined by:  AUMT8820_PEAK_CODE_DOMAIN_ERROR_MEASUREMENT_MODE_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Peak_Code_Domain_Error_Measurement_Mode(ViSession instrSession, ViInt32 mode)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"ON", "OFF", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   if (AuMT8820_InvalidViInt32Range(mode, 0, 1) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   len = sprintf(buf, "PCDE_MEAS %s", commandArray[mode]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Peak Code Domain Error Measurement Mode
       Command(s): PCDE_MEAS[?]
       Purpose:    Set/Get the peak code domain error measurement mode
       Mode defined by:  AUMT8820_PEAK_CODE_DOMAIN_ERROR_MEASUREMENT_MODE_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Peak_Code_Domain_Error_Measurement_Mode(ViSession instrSession, ViInt32  _VI_FAR *mode)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"ON", "OFF", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViChar    resBuf0[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "PCDE_MEAS?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   /* Get if numeric returned from instrument */
   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   /* Scan(rdBuf, "%d", mode);  */

   /* Get if string returned from instrument */
   Scan(rdBuf, "%s", resBuf0); 
   if((auMT8820_status = AuMT8820_StringToEnum(commandArray, resBuf0, mode)) < 0) 
     return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get RMS Phase Vector Error
       Command(s): PHASEERR?
       Purpose:    Gets the most recent phase error measurement.
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_RMS_Phase_Vector_Error_WCDMA(ViSession instrSession, ViReal64 _VI_FAR *degree)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "PHASEERR?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%f", degree);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Downlink PICH
       Command(s): PICH[?]
       Purpose:    Sets the PICH output for downlink channel.
       Setting defined by:  AUMT8820_DOWNLINK_PICH_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Downlink_PICH(ViSession instrSession, ViInt32 setting)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"ON", "OFF", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   if (AuMT8820_InvalidViInt32Range(setting, 0, 1) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   len = sprintf(buf, "PICH %s", commandArray[setting]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Downlink PICH
       Command(s): PICH[?]
       Purpose:    Sets the PICH output for downlink channel.
       Setting defined by:  AUMT8820_DOWNLINK_PICH_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Downlink_PICH(ViSession instrSession, ViInt32  _VI_FAR *setting)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"ON", "OFF", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViChar    resBuf0[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "PICH?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   /* Get if numeric returned from instrument */
   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   /* Scan(rdBuf, "%d", setting); */

   /* Get if string returned from instrument */
   Scan(rdBuf, "%s", resBuf0); 
   if((auMT8820_status = AuMT8820_StringToEnum(commandArray, resBuf0, setting)) < 0) 
     return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Downlink PICH Code
       Command(s): PICHCODE[?]
       Purpose:    Sets/Gets the downlink PICH code value.
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Downlink_PICH_Code(ViSession instrSession, ViInt32 code)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   if (AuMT8820_InvalidViInt32Range(code,0,255) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   len = sprintf(buf, "PICHCODE %d", code);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Downlink PICH Code
       Command(s): PICHCODE[?]
       Purpose:    Sets/Gets the downlink PICH code value.
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Downlink_PICH_Code(ViSession instrSession,  ViInt32 _VI_FAR *code)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "PICHCODE?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%d", code);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Downlink PICH Power
       Command(s): PICHPWR[?]
       Purpose:    Sets the output level for downlink PICH.
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Downlink_PICH_Power(ViSession instrSession, ViReal64 level_dB)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   len = sprintf(buf, "PICHPWR %.1f", level_dB);
   if (AuMT8820_InvalidViReal64Range(level_dB, -30.000000, 0.000000) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Downlink PICH Power
       Command(s): PICHPWR[?]
       Purpose:    Sets the output level for downlink PICH.
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Downlink_PICH_Power(ViSession instrSession, ViReal64 _VI_FAR *level_dB)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "PICHPWR?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%f", level_dB);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get TX Power
       Command(s): POWER?
       Purpose:    Get the transmission power for the last measurement
       Unit defined by:  AUMT8820_TX_POWER_UNIT_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_TX_Power_WCDMA(ViSession instrSession, ViInt32 unit, ViReal64 _VI_FAR *level)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"DBM", "WATT", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "POWER? %s", commandArray[unit]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%f", level);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Uplink PRACH Symbol Rate
       Command(s): PRACHRATE?
       Purpose:    Queries the symbol rate for PRACH.
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Uplink_PRACH_Symbol_Rate(ViSession instrSession, ViReal64 _VI_FAR *rate)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "PRACHRATE?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%f", rate);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Uplink PRACH Scrambling Code
       Command(s): PRACHSCRCODE[?]
       Purpose:    Sets/Gets the uplink PRACH scrambling code value.
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Uplink_PRACH_Scrambling_Code(ViSession instrSession, ViInt32 code)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   if (AuMT8820_InvalidViInt32Range(code,0,15) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   len = sprintf(buf, "PRACHSCRCODE %d", code);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Uplink PRACH Scrambling Code
       Command(s): PRACHSCRCODE[?]
       Purpose:    Sets/Gets the uplink PRACH scrambling code value.
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Uplink_PRACH_Scrambling_Code(ViSession instrSession,  ViInt32 _VI_FAR *code)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "PRACHSCRCODE?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%d", code);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Preamble Retrans Max
       Command(s): PREAMRET[?]
       Purpose:    Sets/Gets the number of preamble  max number for RACH.
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Preamble_Retrans_Max(ViSession instrSession, ViInt32 number)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   if (AuMT8820_InvalidViInt32Range(number,1,64) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   len = sprintf(buf, "PREAMRET %d", number);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Preamble Retrans Max
       Command(s): PREAMRET[?]
       Purpose:    Sets/Gets the number of preamble  max number for RACH.
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Preamble_Retrans_Max(ViSession instrSession,  ViInt32 _VI_FAR *number)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "PREAMRET?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%d", number);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Downlink Primary Scrambling Code
       Command(s): PRISCRCODE[?]
       Purpose:    Sets/Gets the downlink primary scrambling code.
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Downlink_Primary_Scrambling_Code(ViSession instrSession, ViInt32 code)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   if (AuMT8820_InvalidViInt32Range(code,0,511) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   len = sprintf(buf, "PRISCRCODE %d", code);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Downlink Primary Scrambling Code
       Command(s): PRISCRCODE[?]
       Purpose:    Sets/Gets the downlink primary scrambling code.
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Downlink_Primary_Scrambling_Code(ViSession instrSession,  ViInt32 _VI_FAR *code)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "PRISCRCODE?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%d", code);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Peak Vector Error
       Command(s): PVECTERR?
       Purpose:    Queries the peak vector error rate.
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Peak_Vector_Error(ViSession instrSession, ViReal64 _VI_FAR *percent)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "PVECTERR?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%f", percent);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Power Offset P0
       Command(s): PWROFS[?]
       Purpose:    Sets the power offset level P0 for RACH.
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Power_Offset_P0(ViSession instrSession, ViInt32 level_dB)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   len = sprintf(buf, "PWROFS %d", level_dB);
   if (AuMT8820_InvalidViInt32Range(level_dB, 1, 8) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Power Offset P0
       Command(s): PWROFS[?]
       Purpose:    Sets the power offset level P0 for RACH.
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Power_Offset_P0(ViSession instrSession, ViInt32 _VI_FAR *level_dB)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "PWROFS?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%d", level_dB);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Average Count for Power Measurement
       Command(s): PWR_AVG[?]
       Purpose:    Set/Get the average count for power measurements
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Average_Count_for_Power_Measurement(ViSession instrSession, ViInt32 count)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   if (AuMT8820_InvalidViInt32Range(count,1,9999) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   len = sprintf(buf, "PWR_AVG %d", count);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Average Count for Power Measurement
       Command(s): PWR_AVG[?]
       Purpose:    Set/Get the average count for power measurements
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Average_Count_for_Power_Measurement(ViSession instrSession,  ViInt32 _VI_FAR *count)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "PWR_AVG?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%d", count);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Power Measurement Mode WCDMA
       Command(s): PWR_MEAS[?]
       Purpose:    Set/Get the power measurement mode
       Mode defined by:  AUMT8820_POWER_MEASUREMENT_MODE_xxx
       Defined in MT8820 GSM LabWindows Driver.c
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Power_Measurement_Mode_WCDMA(ViSession instrSession, ViInt32 mode)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"ON", "OFF", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   if (AuMT8820_InvalidViInt32Range(mode, 0, 1) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   len = sprintf(buf, "PWR_MEAS %s", commandArray[mode]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Power Measurement Mode WCDMA
       Command(s): PWR_MEAS[?]
       Purpose:    Set/Get the power measurement mode
       Mode defined by:  AUMT8820_POWER_MEASUREMENT_MODE_xxx
       Defined in MT8820 GSM LabWindows Driver.c
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Power_Measurement_Mode_WCDMA(ViSession instrSession, ViInt32  _VI_FAR *mode)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"ON", "OFF", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViChar    resBuf0[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "PWR_MEAS?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   /* Get if numeric returned from instrument */
   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   /* Scan(rdBuf, "%d", mode); */

   /* Get if string returned from instrument */
   Scan(rdBuf, "%s", resBuf0); 
   if((auMT8820_status = AuMT8820_StringToEnum(commandArray, resBuf0, mode)) < 0) 
     return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set RRC Filtering Mode
       Command(s): PWR_RRC[?]
       Purpose:    Gets/Sets whether to perform RRC filtering during power measurement
       Unit defined by:  AUMT8820_FILTERED_POWER_UNIT_xxx

       NOT IMPLEMENTED ON MT8820
*/
/*****************************************************************************/
// ViStatus _VI_FUNC AuMT8820_Set_RRC_Filtering_Mode(ViSession instrSession, ViInt32 mode)
//{
//   ViStatus  auMT8820_status = VI_SUCCESS;
//   static ViString commandArray[] = {"ON", "OFF", 0L};
//   ViInt32   len;
//   ViUInt32  retCnt = 0;
//   ViChar    buf[BUFFER_SIZE + 1];
//   
//   if (AuMT8820_InvalidViInt32Range(mode, 0, 1) == VI_TRUE)
//      return VI_ERROR_OUTOF_RANGE;
//   len = sprintf(buf, "PWR_RRC %s", commandArray[mode]);
//   
//   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
//      return auMT8820_status;
//
//   return auMT8820_status;
//
//}
//

/*****************************************************************************/
/*
       Function:   Get RRC Filtering Mode
       Command(s): PWR_RRC[?]
       Purpose:    Gets/Sets whether to perform RRC filtering during power measurement
       Unit defined by:  AUMT8820_FILTERED_POWER_UNIT_xxx

       NOT IMPLEMENTED ON MT8820
*/
/*****************************************************************************/
//ViStatus _VI_FUNC AuMT8820_Get_RRC_Filtering_Mode(ViSession instrSession, ViInt32  _VI_FAR *mode)
//{
//   ViStatus  auMT8820_status = VI_SUCCESS;
//   static ViString commandArray[] = {"ON", "OFF", 0L};
//   ViInt32   len;
//   ViUInt32  retCnt = 0;
//   ViChar    buf[BUFFER_SIZE + 1];
//   ViChar    rdBuf[BUFFER_SIZE];
//   ViChar    resBuf0[BUFFER_SIZE];
//   ViUInt32  bytesRead;
//   
//   len = sprintf(buf, "PWR_RRC?");
//   
//   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
//      return auMT8820_status;
//
//   /* Get if numeric returned from instrument */
//   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
//      return auMT8820_status;
//   Scan(rdBuf, "%d", mode);
//
//   /* Get if string returned from instrument */
//   /*  Scan(rdBuf, "%s", resBuf0); */
//   /*  if((auMT8820_status = AuMT8820_StringToEnum(commandArray, resBuf0, mode)) < 0) */
//   /*    return auMT8820_status;*/
//
//   return auMT8820_status;
//
//}
//

/*****************************************************************************/
/*
       Function:   Set Register Slot Into List
       Command(s): REGSLOTLIST[?]
       Purpose:    Sets/Gets the slot numbers into the slot power window of the time domain screen.
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Register_Slot_Into_List(ViSession instrSession, ViInt32 slot)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   len = sprintf(buf, "REGSLOTLIST %d", slot);
   if (AuMT8820_InvalidViInt32Range(slot, -15, 164) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Register Slot Into List
       Command(s): REGSLOTLIST[?]
       Purpose:    Sets/Gets the slot numbers into the slot power window of the time domain screen.
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Register_Slot_Into_List(ViSession instrSession, ViInt32 _VI_FAR *slot)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "REGSLOTLIST?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%d", slot);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Register Slot Into List Range
       Command(s): REGSLOTLIST n-m
       Purpose:    Registers slot numbers from the slot power window of the time domain measurement screen.
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Register_Slot_Into_List_Range(ViSession instrSession, ViInt32 slotn, ViInt32 slotm)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   if (AuMT8820_InvalidViInt32Range(slotn,-15,164) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   if (AuMT8820_InvalidViInt32Range(slotm,-15,164) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   if (slotn >= slotm)
      return VI_ERROR_OUTOF_RANGE;
   len = sprintf(buf, "REGSLOTLIST %d-%d", slotn, slotm);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set RF Signal Input Level WCDMA
       Command(s): RFLVL[?],REFLVL[?],ILVL[?]
       Purpose:    Set/Get the RF input signal level
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_RF_Signal_Input_Level_WCDMA(ViSession instrSession, ViReal64 level_dBm)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   if (AuMT8820_InvalidViReal64Range(level_dBm, -65.000, 35.000) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   len = sprintf(buf, "RFLVL %.1f", level_dBm);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get RF Signal Input Level WCDMA
       Command(s): RFLVL[?],REFLVL[?],ILVL[?]
       Purpose:    Set/Get the RF input signal level
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_RF_Signal_Input_Level_WCDMA(ViSession instrSession,  ViReal64 _VI_FAR *level_dBm)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "RFLVL?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%f", level_dBm);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Rx Measurement Status WCDMA
       Command(s): RXMSTAT?
       Purpose:    Get the current reception measurement status
       Unit defined by:  AUMT8820_MEASUREMENT_STATUS_xxx
 */
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Rx_Measurement_Status_WCDMA(ViSession instrSession, ViInt32 _VI_FAR *status)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViChar    resBuf0[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "RXMSTAT?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   /* Get if numeric returned from instrument */
   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%d", status);

   /* Get if string returned from instrument */
   /*  Scan(rdBuf, "%s", resBuf0); */
   /*  if((auMT8820_status = AuMT8820_StringToEnum(commandArray, resBuf0, status)) < 0) */
   /*    return auMT8820_status;*/

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set RX Measure Timeout
       Command(s): RX_TIMEOUT[?]
       Purpose:    Sets/Gets the timeout for reception measurement including bit error rate and block error rate.
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_RX_Measure_Timeout(ViSession instrSession, ViInt32 seconds)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   len = sprintf(buf, "RX_TIMEOUT %d", seconds);
   if (AuMT8820_InvalidViInt32Range(seconds, 1, 60) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get RX Measure Timeout
       Command(s): RX_TIMEOUT[?]
       Purpose:    Sets/Gets the timeout for reception measurement including bit error rate and block error rate.
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_RX_Measure_Timeout(ViSession instrSession, ViInt32 _VI_FAR *seconds)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "RX_TIMEOUT?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%d", seconds);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Downlink SCCPCH
       Command(s): SCCPCH[?]
       Purpose:    Sets the SCCPCH for downlink channel.
       Setting defined by:  AUMT8820_DOWNLINK_SCCPCH_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Downlink_SCCPCH(ViSession instrSession, ViInt32 setting)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"ON", "OFF", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   if (AuMT8820_InvalidViInt32Range(setting, 0, 1) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   len = sprintf(buf, "SCCPCH %s", commandArray[setting]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Downlink SCCPCH
       Command(s): SCCPCH[?]
       Purpose:    Sets the SCCPCH for downlink channel.
       Setting defined by:  AUMT8820_DOWNLINK_SCCPCH_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Downlink_SCCPCH(ViSession instrSession, ViInt32  _VI_FAR *setting)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"ON", "OFF", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViChar    resBuf0[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "SCCPCH?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   /* Get if numeric returned from instrument */
   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   /* Scan(rdBuf, "%d", setting); */

   /* Get if string returned from instrument */
   Scan(rdBuf, "%s", resBuf0); 
   if((auMT8820_status = AuMT8820_StringToEnum(commandArray, resBuf0, setting)) < 0) 
     return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Downlink SCCPCH Code
       Command(s): SCCPCHCODE[?]
       Purpose:    Sets/Gets the channelization code for SCCPCH.
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Downlink_SCCPCH_Code(ViSession instrSession, ViInt32 code)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   if (AuMT8820_InvalidViInt32Range(code,0,63) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   len = sprintf(buf, "SCCPCHCODE %d", code);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Downlink SCCPCH Code
       Command(s): SCCPCHCODE[?]
       Purpose:    Sets/Gets the channelization code for SCCPCH.
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Downlink_SCCPCH_Code(ViSession instrSession,  ViInt32 _VI_FAR *code)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "SCCPCHCODE?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%d", code);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Downlink SCCPCH Power
       Command(s): SCCPCHPWR[?]
       Purpose:    Sets/Gets the SCCPCH power for downlink physical channel.
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Downlink_SCCPCH_Power(ViSession instrSession, ViReal64 level_dB)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   len = sprintf(buf, "SCCPCHPWR %.1f", level_dB);
   if (AuMT8820_InvalidViReal64Range(level_dB, -30.000000, 0.000000) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Downlink SCCPCH Power
       Command(s): SCCPCHPWR[?]
       Purpose:    Sets/Gets the SCCPCH power for downlink physical channel.
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Downlink_SCCPCH_Power(ViSession instrSession, ViReal64 _VI_FAR *level_dB)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "SCCPCHPWR?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%f", level_dB);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Downlink SCCPCH Symbol Rate
       Command(s): SCCPCHRATE?
       Purpose:    Queries the symbol rate for SCCPCH downlink channel.
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Downlink_SCCPCH_Symbol_Rate(ViSession instrSession, ViReal64 _VI_FAR *level)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "SCCPCHRATE?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%f", level);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Downlink SCCPCH Timing Offset
       Command(s): SCCPCHTOFS[?]
       Purpose:    Sets/Gets the SCCPCH timing offset for downlink physical channel.
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Downlink_SCCPCH_Timing_Offset(ViSession instrSession, ViInt32 offset)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   len = sprintf(buf, "SCCPCHTOFS %d", offset);
   if (AuMT8820_InvalidViReal64Range(offset, 0, 149) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Downlink SCCPCH Timing Offset
       Command(s): SCCPCHTOFS[?]
       Purpose:    Sets/Gets the SCCPCH timing offset for downlink physical channel.
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Downlink_SCCPCH_Timing_Offset(ViSession instrSession, ViInt32 _VI_FAR *offset)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "SCCPCHTOFS?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%d", offset);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Downlink SCH
       Command(s): SCH[?]
       Purpose:    Sets the SCH for downlink channel.
       Setting defined by:  AUMT8820_DOWNLINK_SCH_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Downlink_SCH(ViSession instrSession, ViInt32 setting)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"ON", "OFF", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   if (AuMT8820_InvalidViInt32Range(setting, 0, 1) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   len = sprintf(buf, "SCH %s", commandArray[setting]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Downlink SCH
       Command(s): SCH[?]
       Purpose:    Sets the SCH for downlink channel.
       Setting defined by:  AUMT8820_DOWNLINK_SCH_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Downlink_SCH(ViSession instrSession, ViInt32  _VI_FAR *setting)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"ON", "OFF", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViChar    resBuf0[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "SCH?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   /* Get if numeric returned from instrument */
   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   /* Scan(rdBuf, "%d", setting); */

   /* Get if string returned from instrument */
   Scan(rdBuf, "%s", resBuf0); 
   if((auMT8820_status = AuMT8820_StringToEnum(commandArray, resBuf0, setting)) < 0) 
     return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Downlink SCH Power
       Command(s): SCHPWR[?]
       Purpose:    Sets/Gets the SCH power for downlink physical channel.
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Downlink_SCH_Power(ViSession instrSession, ViReal64 level_dB)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   len = sprintf(buf, "SCHPWR %.1f", level_dB);
   if (AuMT8820_InvalidViReal64Range(level_dB, -30.000000, 0.000000) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Downlink SCH Power
       Command(s): SCHPWR[?]
       Purpose:    Sets/Gets the SCH power for downlink physical channel.
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Downlink_SCH_Power(ViSession instrSession, ViReal64 _VI_FAR *level_dB)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "SCHPWR?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%f", level_dB);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Active Window WCDMA
       Command(s): SCRACT[?]
       Purpose:    Switches the active window.
       Window defined by:  AUMT8820_ACTIVE_WINDOW_WCDMA_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Active_Window_WCDMA(ViSession instrSession, ViInt32 window)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"SETP", "FMEAS", "SEQMON", "SPMON", "UEREP", "TDMEAS", "PWRLIST", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   if (AuMT8820_InvalidViInt32Range(window, 0, 6) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   len = sprintf(buf, "SCRACT %s", commandArray[window]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Active Window WCDMA
       Command(s): SCRACT[?]
       Purpose:    Switches the active window.
       Window defined by:  AUMT8820_ACTIVE_WINDOW_WCDMA_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Active_Window_WCDMA(ViSession instrSession, ViInt32  _VI_FAR *window)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"SETP", "FMEAS", "SEQMON", "SPMON", "UEREP", "TDMEAS", "PWRLIST", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViChar    resBuf0[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "SCRACT?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   /* Get if numeric returned from instrument */
   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   /* Scan(rdBuf, "%d", window); */

   /* Get if string returned from instrument */
   Scan(rdBuf, "%s", resBuf0); 
   if((auMT8820_status = AuMT8820_StringToEnum(commandArray, resBuf0, window)) < 0) 
     return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Screen Select WCDMA
       Command(s): SCRSEL[?]
       Purpose:    Selects a screen.
       Screen defined by:  AUMT8820_SCREEN_SELECT_WCDMA_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Screen_Select_WCDMA(ViSession instrSession, ViInt32 screen)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"FMEAS", "TDMEAS", "SEQMON", "SPMON", "SYSCFG", "PRMSAVE", "PRMRCL", "SYSINFO",  0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   if (AuMT8820_InvalidViInt32Range(screen, 0, 7) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   len = sprintf(buf, "SCRSEL %s", commandArray[screen]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Screen Select WCDMA
       Command(s): SCRSEL[?]
       Purpose:    Selects a screen.
       Screen defined by:  AUMT8820_SCREEN_SELECT_WCDMA_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Screen_Select_WCDMA(ViSession instrSession, ViInt32  _VI_FAR *screen)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"FMEAS", "TDMEAS", "SEQMON", "SPMON", "SYSCFG", "PRMSAVE", "PRMRCL", "SYSINFO",  0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViChar    resBuf0[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "SCRSEL?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   /* Get if numeric returned from instrument */
   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   /* Scan(rdBuf, "%d", screen); */

   /* Get if string returned from instrument */
   Scan(rdBuf, "%s", resBuf0); 
   if((auMT8820_status = AuMT8820_StringToEnum(commandArray, resBuf0, screen)) < 0) 
     return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Sequential Output
       Command(s): SEQOUT[?]
       Purpose:    Sets/Gets the cycle, variation level and On/Off sequential output.
       Mode defined by:  AUMT8820_SEQUENTIAL_OUTPUT_MODE_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Sequential_Output(ViSession instrSession, ViInt32 step_dB, ViInt32 length_frames, ViInt32 mode)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"ON", "OFF", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   len = sprintf(buf, "SEQOUT %d,%d,%s", step_dB, length_frames, commandArray[mode]);
   if (AuMT8820_InvalidViInt32Range(mode, 0, 1) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   if (AuMT8820_InvalidViInt32Range(step_dB, -30, 30) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   if (AuMT8820_InvalidViInt32Range(length_frames, 2, 31) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Sequential Output
       Command(s): SEQOUT[?]
       Purpose:    Sets/Gets the cycle, variation level and On/Off sequential output.
       Mode defined by:  AUMT8820_SEQUENTIAL_OUTPUT_MODE_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Sequential_Output(ViSession instrSession, ViInt32 _VI_FAR *step_dB, ViInt32 _VI_FAR *length_frames, ViInt32 _VI_FAR *mode)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"ON", "OFF", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViChar    resBuf0[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "SEQOUT?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%d,%d,%s", step_dB, length_frames, resBuf0); 
   if((auMT8820_status = AuMT8820_StringToEnum(commandArray, resBuf0, mode)) < 0) 
     return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Sequential Output Length
       Command(s): SEQOUTLENG[?]
       Purpose:    Sets/Gets the repeated frame cycle for sequential output.
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Sequential_Output_Length(ViSession instrSession, ViInt32 length_frames)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   len = sprintf(buf, "SEQOUTLENG %d", length_frames);
   if (AuMT8820_InvalidViInt32Range(length_frames, 2, 31) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Sequential Output Length
       Command(s): SEQOUTLENG[?]
       Purpose:    Sets/Gets the repeated frame cycle for sequential output.
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Sequential_Output_Length(ViSession instrSession, ViInt32 _VI_FAR *length_frames)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "SEQOUTLENG?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%d", length_frames);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Sequential Output Frame
       Command(s): SEQOUTSTEP[?]
       Purpose:    Sets/Gets the level variation per frame for sequential output.
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Sequential_Output_Frame(ViSession instrSession, ViInt32 step_dB)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   len = sprintf(buf, "SEQOUTSTEP %d", step_dB);
   if (AuMT8820_InvalidViInt32Range(step_dB, -30, 30) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Sequential Output Frame
       Command(s): SEQOUTSTEP[?]
       Purpose:    Sets/Gets the level variation per frame for sequential output.
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Sequential_Output_Frame(ViSession instrSession, ViInt32 _VI_FAR *step_dB)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "SEQOUTSTEP?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%d", step_dB);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Sequential Output Mode
       Command(s): SEQOUTSW[?]
       Purpose:    Set/Get the sequential output mode
       mode defined by:  AUMT8820_SEQUENTIAL_OUTPUT_MODE__xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Sequential_Output_Mode(ViSession instrSession, ViInt32 mode)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"ON", "OFF", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   if (AuMT8820_InvalidViInt32Range(mode,0,1) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   len = sprintf(buf, "SEQOUTSW %s", commandArray[mode]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Sequential Output Mode
       Command(s): SEQOUTSW[?]
       Purpose:    Set/Get the sequential output mode
       mode defined by:  AUMT8820_SEQUENTIAL_OUTPUT_MODE__xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Sequential_Output_Mode(ViSession instrSession, ViInt32  _VI_FAR *mode)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"ON", "OFF", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViChar    resBuf0[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "SEQOUTSW?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   /* Get if numeric returned from instrument */
   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0) 
      return auMT8820_status;															   
   /* Scan(rdBuf, "%d", mode);  */

   /* Get if string returned from instrument */
   Scan(rdBuf, "%s", resBuf0); 
   if((auMT8820_status = AuMT8820_StringToEnum(commandArray, resBuf0, mode)) < 0) 
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Cell Selection and Reselection Info
       Command(s): SINTRASCH[?]
       Purpose:    Cell selection and Reselection info for SIB3\4   
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Cell_Selection_and_Reselection_Info(ViSession instrSession, ViInt32 level_dB)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   len = sprintf(buf, "SINTRASCH %.d", level_dB);
   if (AuMT8820_InvalidViInt32Range(level_dB, -16, 10) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Cell Selection and Reselection Info
       Command(s): SINTRASCH[?]
       Purpose:    Cell selection and Reselection info for SIB3\4
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Cell_Selection_and_Reselection_Info(ViSession instrSession, ViInt32 _VI_FAR *level_dB)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "SINTRASCH?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%d", level_dB);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Slot List Mode
       Command(s): SLOTLIST[?]
       Purpose:    Sets the display of the slot average power window.
       Setting defined by:  AUMT8820_SLOT_LIST_MODE_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Slot_List_Mode(ViSession instrSession, ViInt32 setting)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"ON", "OFF", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   if (AuMT8820_InvalidViInt32Range(setting, 0, 1) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   len = sprintf(buf, "SLOTLIST %s", commandArray[setting]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Slot List Mode
       Command(s): SLOTLIST[?]
       Purpose:    Sets the display of the slot average power window.
       Setting defined by:  AUMT8820_SLOT_LIST_MODE_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Slot_List_Mode(ViSession instrSession, ViInt32  _VI_FAR *setting)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"ON", "OFF", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViChar    resBuf0[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "SLOTLIST?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   /* Get if numeric returned from instrument */
   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   /* Scan(rdBuf, "%d", setting);*/

   /* Get if string returned from instrument */
   Scan(rdBuf, "%s", resBuf0); 
   if((auMT8820_status = AuMT8820_StringToEnum(commandArray, resBuf0, setting)) < 0) 
     return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Slot List Total
       Command(s): SLOTLIST_TOTAL?
       Purpose:    Queries the total number of registered slots in the slot window.
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Slot_List_Total(ViSession instrSession, ViInt32 _VI_FAR *number)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "SLOTLIST_TOTAL?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%d", number);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Slot Power List
       Command(s): SLOTNO?
       Purpose:    Queries the slot number of registered slots in the slot window.

       NOT IMPLEMENTED  IN MT8820
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Slot_Power_List_WCDMA(ViSession instrSession, ViInt32 slot, ViChar _VI_FAR *slots)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   if(slot == (-16))
     len = sprintf(buf, "SLOTNO?");
   if(slot > (-16))
     len = sprintf(buf, "SLOTNO? %d", slot);
   if (AuMT8820_InvalidViInt32Range(slot, -16, 164) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%s", slot);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Slot Power List ALL
       Command(s): SLOTNO? ALL
       Purpose:    Queries the slot number of all registered slots in the slot window.
*/
/*****************************************************************************/
/*ViStatus _VI_FUNC AuMT8820_Get_Slot_Power_List_ALL(ViSession instrSession, ViChar _VI_FAR *slots)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "SLOTNO? ALL");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%s", slots);

   return auMT8820_status;

}

*/
/****************************************************************************/
/*
       Function:   Get Slot Power List Relative
       Command(s): SLOTNO_REL?
       Purpose:    Queries the relative position of registered slots in the slot window.
       Input: slot = -16 to send no slot number to the instrument
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Slot_Power_List_Relative(ViSession instrSession, ViInt32 slot, ViChar _VI_FAR slots[])
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   if(slot == (-16))
     len = sprintf(buf, "SLOTNO_REL?");
   if(slot > (-16))
     len = sprintf(buf, "SLOTNO_REL? %d", slot);
   if (AuMT8820_InvalidViInt32Range(slot, -16, 164) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%s", slots);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Slot Power List Slot Level
       Command(s): SLOT_PWR?
       Purpose:    Queries the level of registered slots in the slot window.
       Input: slot = -16 to send ALL to the instrument
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Slot_Power_List_Slot_Level(ViSession instrSession, ViInt32 slot, ViChar _VI_FAR level_dBm[])
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   if(slot == (-16))
     len = sprintf(buf, "SLOT_PWR? ALL");
   if(slot > (-16))
     len = sprintf(buf, "SLOT_PWR? %d", slot);
   if (AuMT8820_InvalidViInt32Range(slot, -16, 164) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%s", level_dBm);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Slot Power List Slot Level Relative
       Command(s): SLOT_PWR_REL?
       Purpose:    Queries the level of registered slots relative to the baseslot in the slot window.
       Input: slot = -16 to send ALL to the instrument
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Slot_Power_List_Slot_Level_Relative(ViSession instrSession, ViInt32 slot, ViChar _VI_FAR level_dB[])
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   if(slot == (-16))
     len = sprintf(buf, "SLOT_PWR_REL?");
   if(slot > (-16))
     len = sprintf(buf, "SLOT_PWR_REL? %d", slot);
   if (AuMT8820_InvalidViInt32Range(slot, -16, 164) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%s", level_dB);

   return auMT8820_status;

}

/*****************************************************************************/
/*
       Function:   Get Slot Power List Slot Time
       Command(s): SLOT_TIME?
       Purpose:    Queries the time of registered slots in the slot window.
       Input: slot = -16 to send ALL to the instrument
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Slot_Power_List_Slot_Time(ViSession instrSession, ViInt32 slot, ViChar _VI_FAR time_ms[])
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   if (AuMT8820_InvalidViInt32Range(slot, -16, 164) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   if(slot == (-16))
     len = sprintf(buf, "SLOT_TIME? ALL");
   if(slot > (-16))
     len = sprintf(buf, "SLOT_TIME? %d", slot);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%s", time_ms);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Slot Power List Slot Time Relative
       Command(s): SLOT_TIME_REL?
       Purpose:    Queries the releative time of registered slots in the slot window.
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Slot_Power_List_Slot_Time_Relative(ViSession instrSession, ViInt32 slot, ViChar _VI_FAR time_ms[])
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   if (AuMT8820_InvalidViInt32Range(slot, -16, 164) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   if(slot == (-16))
     len = sprintf(buf, "SLOT_TIME_REL?");
   if(slot > (-16))
     len = sprintf(buf, "SLOT_TIME_REL? %d", slot);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%s", time_ms);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Spectrum Emission Mask Template Judgment
       Command(s): SMASKPASS?
       Purpose:    Get the spectrum emission mask template judgement
       Judgement defined by:  AUMT8820_SPECTRUM_EMISSION_MASK_TEMPLATE_JUDGMENT_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Spectrum_Emission_Mask_Template_Judgment(ViSession instrSession, ViInt32 _VI_FAR *judgement)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"PASS", "FAIL", "-", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViChar    resBuf0[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "SMASKPASS?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   /* Get if numeric returned from instrument */
   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   /* Scan(rdBuf, "%d", judgement); */

   /* Get if string returned from instrument */
   Scan(rdBuf, "%s", resBuf0); 
   if((auMT8820_status = AuMT8820_StringToEnum(commandArray, resBuf0, judgement)) < 0) 
     return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Spectrum Emission Mask Limit
       Command(s): SMASK_ADDLIMIT[?]
       Purpose:    Sets an additional limit for the spectrum emission mask measurement.
       Setting defined by:  AUMT8820_SPECTRUM_EMISSION_MASK_LIMIT_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Spectrum_Emission_Mask_Limit(ViSession instrSession, ViInt32 setting)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"NON", "WCDMA1900", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   if (AuMT8820_InvalidViInt32Range(setting, 0, 1) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   len = sprintf(buf, "SMASK_ADDLIMIT %s", commandArray[setting]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Spectrum Emission Mask Limit
       Command(s): SMASK_ADDLIMIT[?]
       Purpose:    Sets an additional limit for the spectrum emission mask measurement.
       Setting defined by:  AUMT8820_SPECTRUM_EMISSION_MASK_LIMIT_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Spectrum_Emission_Mask_Limit(ViSession instrSession, ViInt32  _VI_FAR *setting)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"NON", "WCDMA1900", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViChar    resBuf0[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "SMASK_ADDLIMIT?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   /* Get if numeric returned from instrument */
   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   /* Scan(rdBuf, "%d", setting); */

   /* Get if string returned from instrument */
   Scan(rdBuf, "%s", resBuf0); 
   if((auMT8820_status = AuMT8820_StringToEnum(commandArray, resBuf0, setting)) < 0) 
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Average Count for Spectrum Emission Mask Measurement
       Command(s): SMASK_AVG[?]
       Purpose:    Set/Get the average count for the spectrum emission mask measurements
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Average_Count_for_Spectrum_Emission_Mask_Measurement(ViSession instrSession, ViInt32 count)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   if (AuMT8820_InvalidViInt32Range(count,1,9999) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   len = sprintf(buf, "SMASK_AVG %d", count);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Average Count for Spectrum Emission Mask Measurement
       Command(s): SMASK_AVG[?]
       Purpose:    Set/Get the average count for the spectrum emission mask measurements
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Average_Count_for_Spectrum_Emission_Mask_Measurement(ViSession instrSession,  ViInt32 _VI_FAR *count)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "SMASK_AVG?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%d", count);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Spectrum Emission Detection Mode
       Command(s): SMASK_DET[?]
       Purpose:    Sets the detection mode for spectrum emission mask measurement.
       Mode defined by:  AUMT8820_SPECTRUM_EMISSION_DETECTION_MODE_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Spectrum_Emission_Detection_Mode(ViSession instrSession, ViInt32 mode)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"POS", "NEG", "SAMPLE", "AVG", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   if (AuMT8820_InvalidViInt32Range(mode, 0, 3) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   len = sprintf(buf, "SMASK_DET %s", commandArray[mode]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Spectrum Emission Detection Mode
       Command(s): SMASK_DET[?]
       Purpose:    Sets the detection mode for spectrum emission mask measurement.
       Mode defined by:  AUMT8820_SPECTRUM_EMISSION_DETECTION_MODE_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Spectrum_Emission_Detection_Mode(ViSession instrSession, ViInt32  _VI_FAR *mode)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"POS", "NEG", "SAMPLE", "AVG", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViChar    resBuf0[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "SMASK_DET?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   /* Get if numeric returned from instrument */
   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   /* Scan(rdBuf, "%d", mode); */

   /* Get if string returned from instrument */
   Scan(rdBuf, "%s", resBuf0); 
   if((auMT8820_status = AuMT8820_StringToEnum(commandArray, resBuf0, mode)) < 0) 
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Spectrum Emission Mask Measurement Mode
       Command(s): SMASK_MEAS[?]
       Purpose:    Set/Get the spectrum emission mask measurement mode
       Mode defined by:  AUMT8820_SPECTRUM_EMISSION_MASK_MEASUREMENT_MODE_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Spectrum_Emission_Mask_Measurement_Mode(ViSession instrSession, ViInt32 mode)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"ON", "OFF", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   if (AuMT8820_InvalidViInt32Range(mode, 0, 1) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   len = sprintf(buf, "SMASK_MEAS %s", commandArray[mode]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Spectrum Emission Mask Measurement Mode
       Command(s): SMASK_MEAS[?]
       Purpose:    Set/Get the spectrum emission mask measurement mode
       Mode defined by:  AUMT8820_SPECTRUM_EMISSION_MASK_MEASUREMENT_MODE_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Spectrum_Emission_Mask_Measurement_Mode(ViSession instrSession, ViInt32  _VI_FAR *mode)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"ON", "OFF", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViChar    resBuf0[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "SMASK_MEAS?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   /* Get if numeric returned from instrument */
   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   /* Scan(rdBuf, "%d", mode); */

   /* Get if string returned from instrument */
   Scan(rdBuf, "%s", resBuf0); 
   if((auMT8820_status = AuMT8820_StringToEnum(commandArray, resBuf0, mode)) < 0) 
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Single Sweep
       Command(s): S2,SNGLS
       Purpose:    Starts a measurement in single mode.
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Single_Sweep(ViSession instrSession)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   len = sprintf(buf, "S2");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Spectrum Monitor Detection Mode
       Command(s): SPMDETECT[?]
       Purpose:    Sets the detection mode for spectrum monitor.
       Mode defined by:  AUMT8820_SPECTRUM_MONITOR_DETECTION_MODE_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Spectrum_Monitor_Detection_Mode(ViSession instrSession, ViInt32 mode)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"PEAK", "RMS", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   if (AuMT8820_InvalidViInt32Range(mode, 0, 1) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   len = sprintf(buf, "SPMDETECT %s", commandArray[mode]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Spectrum Monitor Detection Mode
       Command(s): SPMDETECT[?]
       Purpose:    Sets the detection mode for spectrum monitor.
       Mode defined by:  AUMT8820_SPECTRUM_MONITOR_DETECTION_MODE_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Spectrum_Monitor_Detection_Mode(ViSession instrSession, ViInt32  _VI_FAR *mode)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"PEAK", "RMS", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViChar    resBuf0[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "SPMDETECT?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   /* Get if numeric returned from instrument */
   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   /* Scan(rdBuf, "%d", mode); */

   /* Get if string returned from instrument */
   Scan(rdBuf, "%s", resBuf0); 
   if((auMT8820_status = AuMT8820_StringToEnum(commandArray, resBuf0, mode)) < 0) 
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Spectrum Monitor Display Offset
       Command(s): SPMDSPLOFS[?]
       Purpose:    Sets/Gets the waveform display offset for spectrum monitor.
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Spectrum_Monitor_Display_Offset(ViSession instrSession, ViInt32 level_dB)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   len = sprintf(buf, "SPMDSPLOFS %d", level_dB);
   if (AuMT8820_InvalidViInt32Range(level_dB, -10, 10) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Spectrum Monitor Display Offset
       Command(s): SPMDSPLOFS[?]
       Purpose:    Sets/Gets the waveform display offset for spectrum monitor.
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Spectrum_Monitor_Display_Offset(ViSession instrSession, ViInt32 _VI_FAR *level_dB)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "SPMDSPLOFS?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%d", level_dB);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Spectrum Monitor Frequency Span
       Command(s): SPMSPAN[?]
       Purpose:    Sets/Gets the frequency span for spectrum monitor.
       Span defined by:  AUMT8820_SPECTRUM_MONITOR_FREQUENCY_SPAN_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Spectrum_Monitor_Frequency_Span(ViSession instrSession, ViInt32 span)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"5", "25", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   if (AuMT8820_InvalidViInt32Range(span, 0, 1) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   len = sprintf(buf, "SPMSPAN %s", commandArray[span]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Spectrum Monitor Frequency Span
       Command(s): SPMSPAN[?]
       Purpose:    Sets/Gets the frequency span for spectrum monitor.
       Span defined by:  AUMT8820_SPECTRUM_MONITOR_FREQUENCY_SPAN_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Spectrum_Monitor_Frequency_Span(ViSession instrSession, ViInt32 _VI_FAR *span)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"5", "25", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViChar    resBuf0[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "SPMSPAN?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   /* Get if numeric returned from instrument */
   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   /* Scan(rdBuf, "%d", span); */

   /* Get if string returned from instrument */
   Scan(rdBuf, "%s", resBuf0); 
   if((auMT8820_status = AuMT8820_StringToEnum(commandArray, resBuf0, span)) < 0) 
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Synchronous Single Sweep
       Command(s): SWP[?],TS
       Purpose:    Synchronous single sweep

*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Synchronous_Single_Sweep_WCDMA(ViSession instrSession)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"0", "1" ,0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViChar    resBuf0[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "SWP;SWP?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

//   /* Get if numeric returned from instrument */
//   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
//      return auMT8820_status;
//   Scan(rdBuf, "%d", status);
//
   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Synchronous Single Sweep
       Command(s): SWP[?],TS
       Purpose:    Synchronous single sweep
       Status defined by:  AUMT8820__xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Synchronous_Single_Sweep_WCDMA(ViSession instrSession, ViInt32  _VI_FAR *status)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"0", "1" ,0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViChar    resBuf0[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "SWP?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   /* Get if numeric returned from instrument */
   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
     return auMT8820_status;
  Scan(rdBuf, "%d", status);

   /* Get if string returned from instrument */
   /*  Scan(rdBuf, "%s", resBuf0); */
   /*  if((auMT8820_status = AuMT8820_StringToEnum(commandArray, resBuf0, status)) < 0) */
   /*    return auMT8820_status;*/

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Sweep and Page
       Command(s): SWPANDPG
       Purpose:    Performs paging after sweep.
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Sweep_and_Page(ViSession instrSession)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   len = sprintf(buf, "SWPANDPG");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Sync Marker 1 and 2
       Command(s): SYNCMKR[?]
       Purpose:    Sets the inactive marker synchronously with active one when two markers are displayed.
       Setting defined by:  AUMT8820_SYNC_MARKER_1_AND_2_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Sync_Marker_1_and_2(ViSession instrSession, ViInt32 setting)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"ON", "OFF", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   if (AuMT8820_InvalidViInt32Range(setting, 0, 1) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   len = sprintf(buf, "SYNCMKR %s", commandArray[setting]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Sync Marker 1 and 2
       Command(s): SYNCMKR[?]
       Purpose:    Sets the inactive marker synchronously with active one when two markers are displayed.
       Setting defined by:  AUMT8820_SYNC_MARKER_1_AND_2_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Sync_Marker_1_and_2(ViSession instrSession, ViInt32  _VI_FAR *setting)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"ON", "OFF", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViChar    resBuf0[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "SYNCMKR?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   /* Get if numeric returned from instrument */
   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   /* Scan(rdBuf, "%d", setting); */

   /* Get if string returned from instrument */
   Scan(rdBuf, "%s", resBuf0); 
   if((auMT8820_status = AuMT8820_StringToEnum(commandArray, resBuf0, setting)) < 0) 
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Tag Select
       Command(s): TAGSEL[?]
       Purpose:    Displays a tag in the current screen or window on top to activate it.
       Window defined by:  AUMT8820_WINDOW_SELECT_WCDMA_xxx
       Tag defined by:  AUMT8820_TAG_WCDMA_xxx_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Tag_Select_WCDMA(ViSession instrSession, ViInt32 tag)
{
  ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandtagArray[] = {"SETP,COM", "SETP,PHYCH", "SETP,CALL", "SETP,TXMEAS", "SETP,RXMEAS","SETP,FMEAS","SETP,TDMEAS", "SETP,SPMON", "SETP,AUDIO", "FMEAS,PWR", "FMEAS,FREQ", "FMEAS,OBW", "FMEAS,SMASK", "FMEAS,ADJ", "FMEAS,MOD", "FMEAS,PCDE", "FMEAS,BER", "FMEAS,BLER","UEREP,UEREP", "SYSCFG,COMMON", "SYSCFG,PHONE1", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   len = sprintf(buf, "TAGSEL %s",commandtagArray[tag]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Tag Select
       Command(s): TAGSEL[?]
       Purpose:    Displays a tag in the current screen or window on top to activate it.
       Window defined by:  AUMT8820_WINDOW_SELECT_WCDMA_xxx
       Tag defined by:  AUMT8820_TAG_WCDMA_xxx_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Tag_Select_WCDMA(ViSession instrSession, ViInt32 window, ViInt32  _VI_FAR *tag)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"SETP", "FMEAS", "UEREP", "SYSCFG", 0L};
   static ViString commandArray2[] = {"COM", "PHYCH", "CALL", "TXMEAS", "RXMEAS", "FMEAS", "TDMEAS", "SPMON", "AUDIO",
                                     "PWR", "FREQ", "OBW", "SMASK", "ADJ", "MOD", "PCDE", "BER", "BLER",
                                     "UEREP",
                                     "COMMON", "PHONE1", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViChar    resBuf0[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "TAGSEL? %s", commandArray[window]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   /* Get if numeric returned from instrument */
   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   /* Scan(rdBuf, "%d", tag); */

   /* Get if string returned from instrument */
   Scan(rdBuf, "%s", resBuf0); 
   if((auMT8820_status = AuMT8820_StringToEnum(commandArray2, resBuf0, tag)) < 0) 
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Display Offset
       Command(s): TDMDSPLOFS[?]
       Purpose:    Sets/Gets the offset for reference level and sets it as the upper limit value of the measured result graph during time domain measurement.
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Display_Offset(ViSession instrSession, ViInt32 offset_dB)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   len = sprintf(buf, "TDMDSPLOFS %dDB", offset_dB);
   if (AuMT8820_InvalidViInt32Range(offset_dB, -10, 10) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Display Offset
       Command(s): TDMDSPLOFS[?]
       Purpose:    Sets/Gets the offset for reference level and sets it as the upper limit value of the measured result graph during time domain measurement.
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Display_Offset(ViSession instrSession, ViInt32 _VI_FAR *offset_dB)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "TDMDSPLOFS?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%d", offset_dB);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set RRC Filter
       Command(s): TDM_RRC[?]
       Purpose:    Sets whether to perform RRC filtering during time domain measurement.
       Setting defined by:  AUMT8820_TIME_DOMAIN_RRC_FILTER_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_RRC_Filter(ViSession instrSession, ViInt32 setting)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"ON", "OFF", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   if (AuMT8820_InvalidViInt32Range(setting, 0, 1) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   len = sprintf(buf, "TDM_RRC %s", commandArray[setting]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get RRC Filter
       Command(s): TDM_RRC[?]
       Purpose:    Sets whether to perform RRC filtering during time domain measurement.
       Setting defined by:  AUMT8820_TIME_DOMAIN_RRC_FILTER_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_RRC_Filter(ViSession instrSession, ViInt32  _VI_FAR *setting)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"ON", "OFF", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViChar    resBuf0[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "TDM_RRC?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   /* Get if numeric returned from instrument */
   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   /* Scan(rdBuf, "%d", setting); */

   /* Get if string returned from instrument */
   Scan(rdBuf, "%s", resBuf0); 
   if((auMT8820_status = AuMT8820_StringToEnum(commandArray, resBuf0, setting)) < 0) 
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Template Close WCDMA
       Command(s): TEMPCLS
       Purpose:    Closes the template in the fundamental measurement screen.
       Template defined by:  AUMT8820_TEMPLATE_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Template_Close_WCDMA(ViSession instrSession, ViInt32  template)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"SMASK", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
//   if (AuMT8820_InvalidViInt32Range(template, 0, 1) == VI_TRUE)
   if (template == 0)
      auMT8820_status = VI_SUCCESS;
   else
      return VI_ERROR_OUTOF_RANGE;
   len = sprintf(buf, "TEMPCLS %s", commandArray[template]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Spectrum Emission Mask Template Offset
       Command(s): TEMPLVL_SMASK[?]
       Purpose:    Sets the offset frequency level at the spectrum emission mask template.
       Offset defined by:  AUMT8820_SPECTRUM_EMISSION_MASK_TEMPLATE_OFFSET_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Spectrum_Emission_Mask_Template_Offset(ViSession instrSession, ViInt32 offset, ViReal64 level_dBc)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"1", "2", "3", "4", "5", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   if (AuMT8820_InvalidViReal64Range(level_dBc, -100.00, 0.00) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   if (AuMT8820_InvalidViInt32Range(offset, 0, 4) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   len = sprintf(buf, "TEMPLVL_SMASK %s,%.1f", commandArray[offset], level_dBc);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Spectrum Emission Mask Template Offset
       Command(s): TEMPLVL_SMASK[?]
       Purpose:    Sets the offset frequency level at the spectrum emission mask template.
       Offset defined by:  AUMT8820_SPECTRUM_EMISSION_MASK_TEMPLATE_OFFSET_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Spectrum_Emission_Mask_Template_Offset(ViSession instrSession, ViInt32 offset, ViReal64  _VI_FAR *level_dBc)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"1", "2", "3", "4", "5", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViChar    resBuf0[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "TEMPLVL_SMASK? %s", commandArray[offset]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   /* Get if numeric returned from instrument */
   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%f", level_dBc);

   /* Get if string returned from instrument */
   /*  Scan(rdBuf, "%s", resBuf0); */
   /*  if((auMT8820_status = AuMT8820_StringToEnum(commandArray, resBuf0, offset)) < 0) */
   /*    return auMT8820_status;*/

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Template Open
       Command(s): TEMPOPN[?]
       Purpose:    Opens the template in the fundamental measurement screen.
       Template defined by:  AUMT8820_TEMPLATE_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Template_Open_WCDMA(ViSession instrSession, ViInt32 template)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"SMASK", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
//   if (AuMT8820_InvalidViInt32Range(template, 0, 1) == VI_TRUE)
   if (template == 0)
      auMT8820_status = VI_SUCCESS;
   else
      return VI_ERROR_OUTOF_RANGE;
   len = sprintf(buf, "TEMPOPN %s", commandArray[template]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Template Open
       Command(s): TEMPOPN[?]
       Purpose:    Opens the template in the fundamental measurement screen.
       Template defined by:  AUMT8820_TEMPLATE_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Template_Open_WCDMA(ViSession instrSession, ViInt32  _VI_FAR *template)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"SMASK", "PWR", "NONE", "", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViChar    resBuf0[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "TEMPOPN?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   /* Get if numeric returned from instrument */
   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   /* Scan(rdBuf, "%d", template); */

   /* Get if string returned from instrument */
   Scan(rdBuf, "%s", resBuf0); 
   if((auMT8820_status = AuMT8820_StringToEnum(commandArray, resBuf0, template)) < 0) 
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Loopback Mode
       Command(s): TESTMODE[?]
       Purpose:    Set/Get the test loop mode for call processing
       Mode defined by:  AUMT8820_LOOPBACK_MODE_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Loopback_Mode(ViSession instrSession, ViInt32 mode)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"OFF", "MODE1", "MODE2", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   if (AuMT8820_InvalidViInt32Range(mode, 0, 2) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   len = sprintf(buf, "TESTMODE %s", commandArray[mode]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Loopback Mode
       Command(s): TESTMODE[?]
       Purpose:    Set/Get the test loop mode for call processing
       Mode defined by:  AUMT8820_LOOPBACK_MODE_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Loopback_Mode(ViSession instrSession, ViInt32  _VI_FAR *mode)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"OFF", "MODE1", "MODE2", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViChar    resBuf0[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "TESTMODE?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   /* Get if numeric returned from instrument */
   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   /* Scan(rdBuf, "%d", mode); */

   /* Get if string returned from instrument */
   Scan(rdBuf, "%s", resBuf0); 
   if((auMT8820_status = AuMT8820_StringToEnum(commandArray, resBuf0, mode)) < 0) 
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Timeout Length
       Command(s): TIMEOUT[?]
       Purpose:    Sets/Gets the timeout period for time domain measurement.
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Timeout_Length(ViSession instrSession, ViInt32 seconds)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   len = sprintf(buf, "TIMEOUT %dS", seconds);
   if (AuMT8820_InvalidViInt32Range(seconds, 1, 60) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Timeout Length
       Command(s): TIMEOUT[?]
       Purpose:    Sets/Gets the timeout period for time domain measurement.
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Timeout_Length(ViSession instrSession, ViInt32 _VI_FAR *seconds)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "TIMEOUT?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%d", seconds);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Time Span
       Command(s): TIMSPAN[?]
       Purpose:    Sets/Gets the time span for time domain measurement.
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Time_Span(ViSession instrSession, ViReal64 milliseconds)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   len = sprintf(buf, "TIMSPAN %.1fMS", milliseconds);
   if (AuMT8820_InvalidViReal64Range(milliseconds, 1.000000, 1000.000000) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Time Span
       Command(s): TIMSPAN[?]
       Purpose:    Sets/Gets the time span for time domain measurement.
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Time_Span(ViSession instrSession, ViReal64 _VI_FAR *milliseconds)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "TIMSPAN?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%f", milliseconds);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Timing Error
       Command(s): TMGERR?
       Purpose:    Queries the result for the most recent timing error measurement.
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Timing_Error(ViSession instrSession, ViReal64 _VI_FAR *chips)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "TMGERR?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%f", chips);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Power Control Algorithm
       Command(s): TPCALGO[?]
       Purpose:    Sets the inner loop power control algorithm.
       Algorithm defined by:  AUMT8820_POWER_CONTROL_ALGORITHM_x
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Power_Control_Algorithm(ViSession instrSession, ViInt32 algorithm)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"1", "2", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   if (AuMT8820_InvalidViInt32Range(algorithm, 0, 1) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   len = sprintf(buf, "TPCALGO %s", commandArray[algorithm]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Power Control Algorithm
       Command(s): TPCALGO[?]
       Purpose:    Sets the inner loop power control algorithm.
       Algorithm defined by:  AUMT8820_POWER_CONTROL_ALGORITHM_x
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Power_Control_Algorithm(ViSession instrSession, ViInt32  _VI_FAR *algorithm)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"1", "2", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViChar    resBuf0[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "TPCALGO?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   /* Get if numeric returned from instrument */
   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   /* Scan(rdBuf, "%d", algorithm); */

   /* Get if string returned from instrument */
   Scan(rdBuf, "%s", resBuf0); 
   if((auMT8820_status = AuMT8820_StringToEnum(commandArray, resBuf0, algorithm)) < 0) 
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Power Control Bit Pattern
       Command(s): TPCPAT[?]
       Purpose:    Sets the power control bit pattern.
       Pattern defined by:  AUMT8820_POWER_CONTROL_BIT_PATTERN_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Power_Control_Bit_Pattern(ViSession instrSession, ViInt32 pattern)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"ALL0", "ALL1", "ALT", "ILPC", "UCMD", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   if (AuMT8820_InvalidViInt32Range(pattern, 0, 4) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   len = sprintf(buf, "TPCPAT %s", commandArray[pattern]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Power Control Bit Pattern
       Command(s): TPCPAT[?]
       Purpose:    Sets the power control bit pattern.
       Pattern defined by:  AUMT8820_POWER_CONTROL_BIT_PATTERN_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Power_Control_Bit_Pattern(ViSession instrSession, ViInt32  _VI_FAR *pattern)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"ALL0", "ALL1", "ALT", "ILPC", "UCMD", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViChar    resBuf0[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "TPCPAT?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   /* Get if numeric returned from instrument */
   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   /* Scan(rdBuf, "%d", pattern); */

   /* Get if string returned from instrument */
   Scan(rdBuf, "%s", resBuf0); 
   if((auMT8820_status = AuMT8820_StringToEnum(commandArray, resBuf0, pattern)) < 0) 
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set TPC Step Size
       Command(s): TPCSTEP[?]
       Purpose:    Sets the TPC step size.
       Unit defined by:  AUMT8820_FILTERED_POWER_UNIT_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_TPC_Step_Size(ViSession instrSession, ViInt32 step)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"1", "2", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   if (AuMT8820_InvalidViInt32Range(step, 0, 1) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   len = sprintf(buf, "TPCSTEP %s", commandArray[step]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get TPC Step Size
       Command(s): TPCSTEP[?]
       Purpose:    Sets the TPC step size.
       Unit defined by:  AUMT8820_FILTERED_POWER_UNIT_xxx
 */
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_TPC_Step_Size(ViSession instrSession, ViInt32  _VI_FAR *step)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"1", "2", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViChar    resBuf0[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "TPCSTEP?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   /* Get if numeric returned from instrument */
   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   /* Scan(rdBuf, "%d", step); */

   /* Get if string returned from instrument */
   Scan(rdBuf, "%s", resBuf0); 
   if((auMT8820_status = AuMT8820_StringToEnum(commandArray, resBuf0, step)) < 0) 
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set TPC User Command
       Command(s): TPCUCMD[?]
       Purpose:    Sets the user defined command for power control.
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_TPC_User_Command(ViSession instrSession, ViInt32 part, ViChar *command)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   if (AuMT8820_InvalidViInt32Range(part,1,6) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
//   if (AuMT8820_InvalidViInt32Range(command,0,31) == VI_TRUE)
//      return VI_ERROR_OUTOF_RANGE;
   len = sprintf(buf, "TPCUCMD %d,%s", part, command);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get TPC User Command
       Command(s): TPCUCMD[?]
       Purpose:    Sets the user defined command for power control.
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_TPC_User_Command(ViSession instrSession, ViInt32 part, ViChar *command)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   if (AuMT8820_InvalidViInt32Range(part,1,6) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   len = sprintf(buf, "TPCUCMD? %d", part);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%s", command);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set TPC User Command All
       Command(s): TPCUCMDALL[?]
       Purpose:    Sets the user defined command for power control.
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_TPC_User_Command_All(ViSession instrSession, ViChar *command)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
//   if (AuMT8820_InvalidViInt32Range(command[0],0,31) == VI_TRUE)
//      return VI_ERROR_OUTOF_RANGE;
//   if (AuMT8820_InvalidViInt32Range(command[1],0,31) == VI_TRUE)
//      return VI_ERROR_OUTOF_RANGE;
//   if (AuMT8820_InvalidViInt32Range(command[2],0,31) == VI_TRUE)
//      return VI_ERROR_OUTOF_RANGE;
//   if (AuMT8820_InvalidViInt32Range(command[3],0,31) == VI_TRUE)
//      return VI_ERROR_OUTOF_RANGE;
//   if (AuMT8820_InvalidViInt32Range(command[4],0,31) == VI_TRUE)
//      return VI_ERROR_OUTOF_RANGE;
//   if (AuMT8820_InvalidViInt32Range(command[5],0,31) == VI_TRUE)
//      return VI_ERROR_OUTOF_RANGE;
   len = sprintf(buf, "TPCUCMDALL %s", command);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get TPC User Command All
       Command(s): TPCUCMDALL[?]
       Purpose:    Sets the user defined command for power control.
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_TPC_User_Command_All(ViSession instrSession,  ViChar *command)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "TPCUCMDALL?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%s", command);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set AICH Transmission Timing
       Command(s): TRANSTIMING[?]
       Purpose:    Sets the AICH transmission timing.
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_AICH_Transmission_Timing(ViSession instrSession, ViInt32 number)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   if (AuMT8820_InvalidViInt32Range(number,0,1) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   len = sprintf(buf, "TRANSTIMING %d", number);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get AICH Transmission Timing
       Command(s): TRANSTIMING[?]
       Purpose:    Sets the AICH transmission timing.
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_AICH_Transmission_Timing(ViSession instrSession,  ViInt32 _VI_FAR *number)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "TRANSTIMING?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%d", number);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Trigger Delay
       Command(s): TRGDELAY[?]
       Purpose:    Sets/Gets the delay time from trigger to start of sweep.
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Trigger_Delay(ViSession instrSession, ViReal64 milliseconds)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   len = sprintf(buf, "TRGDELAY %.4fMS", milliseconds);
   if (AuMT8820_InvalidViReal64Range(milliseconds, -10.000000, 10.000000) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Trigger Delay
       Command(s): TRGDELAY[?]
       Purpose:    Sets/Gets the delay time from trigger to start of sweep.
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Trigger_Delay(ViSession instrSession, ViReal64 _VI_FAR *milliseconds)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "TRGDELAY?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%f", milliseconds);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Trigger Source
       Command(s): TRGSRC[?]
       Purpose:    Sets the trigger for time domain measurement.
       Trigger defined by:  AUMT8820_TRIGGER_SOURCE_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Trigger_Source(ViSession instrSession, ViInt32 trigger)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"VIDEO", "RISEVIDEO", "FALLVIDEO","INT", "EXT","FREE",   0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   if (AuMT8820_InvalidViInt32Range(trigger, 0, 5) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   len = sprintf(buf, "TRGSRC %s", commandArray[trigger]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Trigger Source
       Command(s): TRGSRC[?]
       Purpose:    Sets the trigger for time domain measurement.
       Trigger defined by:  AUMT8820_TRIGGER_SOURCE_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Trigger_Source(ViSession instrSession, ViInt32  _VI_FAR *trigger)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] ={"VIDEO", "RISEVIDEO", "FALLVIDEO","INT", "EXT","FREE",   0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViChar    resBuf0[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "TRGSRC?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   /* Get if numeric returned from instrument */
   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   /* Scan(rdBuf, "%d", trigger); */

   /* Get if string returned from instrument */
   Scan(rdBuf, "%s", resBuf0); 
   if((auMT8820_status = AuMT8820_StringToEnum(commandArray, resBuf0, trigger)) < 0) 
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Carrier Frequency Error Total WCDMA
       Command(s): TTL_CARRFERR?
       Purpose:    Gets the total value among the average count of frequency error measurement.
       Unit defined by:  AUMT8820_CARRIER_FREQ_ERROR_UNIT_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Carrier_Frequency_Error_Total_WCDMA(ViSession instrSession, ViInt32  unit, ViReal64  _VI_FAR *average, ViReal64  _VI_FAR *maximum, ViReal64  _VI_FAR *minimum)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"HZ", "PPM", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViChar    resBuf0[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   if (AuMT8820_InvalidViInt32Range(unit, 0, 1) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   len = sprintf(buf, "TTL_CARRFERR? %s", commandArray[unit]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   /* Get if numeric returned from instrument */
   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%f,%f,%f", average, maximum, minimum);

   /* Get if string returned from instrument */
   /*  Scan(rdBuf, "%s", resBuf0); */
   /*  if((auMT8820_status = AuMT8820_StringToEnum(commandArray, resBuf0, unit)) < 0) */
   /*    return auMT8820_status;*/

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Error Vector Magnitude Total
       Command(s): TTL_EVM?
       Purpose:    Queries the average, maximum, and minimum values among measured results for EVM.
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Error_Vector_Magnitude_Total(ViSession instrSession, ViReal64 _VI_FAR *avg_percent, ViReal64 _VI_FAR *max_percent, ViReal64 _VI_FAR *min_percent)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "TTL_EVM?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%f,%f,%f", avg_percent, max_percent, min_percent);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Filtered Power Total
       Command(s): TTL_FILTPWR?
       Purpose:    Queries the average, maximum, and minimum values among measured results for RRC filtered power.
       Unit defined by:  AUMT8820_FILTERED_POWER_UNIT_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Filtered_Power_Total(ViSession instrSession, ViInt32  unit, ViReal64  _VI_FAR *average, ViReal64  _VI_FAR *maximum, ViReal64  _VI_FAR *minimum)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"DBM", "WATT", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   if (AuMT8820_InvalidViInt32Range(unit, 0, 1) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   len = sprintf(buf, "TTL_FILTPWR? %s", commandArray[unit]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%f,%f,%f", average, maximum, minimum);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get IQ Imbalance Total
       Command(s): TTL_IQIMB?
       Purpose:    Queries the average, maximum, and minimum values among measured results for IQ imbalance.
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_IQ_Imbalance_Total(ViSession instrSession, ViReal64 _VI_FAR *avg_percent, ViReal64 _VI_FAR *max_percent, ViReal64 _VI_FAR *min_percent)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "TTL_IQIMB?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%f,%f,%f", avg_percent, max_percent, min_percent);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Magnitude Error Total WCDMA
       Command(s): TTL_MAGTDERR?
       Purpose:    Queries the total value of magnitude error measurement results.
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Magnitude_Error_Total_WCDMA(ViSession instrSession, ViReal64 _VI_FAR *avg_percent, ViReal64 _VI_FAR *max_percent, ViReal64 _VI_FAR *min_percent)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "TTL_MAGTDERR?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%f,%f,%f", avg_percent, max_percent, min_percent);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Adjacent Channel Power Total
       Command(s): TTL_MODPWR?
       Purpose:    Queries the average, maximum, and minimum values among measured results for adjacent channel leakage power.
       Offset defined by:  AUMT8820_ADJACENT_CHAN_POWER_OFFSET_xxx
       Unit defined by:  AUMT8820_ADJACENT_CHAN_POWER_UNIT_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Adjacent_Channel_Power_Total(ViSession instrSession, ViInt32 offset, ViInt32 unit, ViReal64 _VI_FAR *avg_level, ViReal64 _VI_FAR *max_level, ViReal64 _VI_FAR *min_level)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"LOW10", "LOW5", "UP5", "UP10", 0L};
   static ViString commandArray2[] = {"DB", "DBM", "WATT", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   if (AuMT8820_InvalidViInt32Range(offset, 0, 3) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   if (AuMT8820_InvalidViInt32Range(unit, 0, 2) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   len = sprintf(buf, "TTL_MODPWR? %s,%s", commandArray[offset], commandArray2[unit]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%f,%f,%f", avg_level, max_level, min_level);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Origin Offset Total
       Command(s): TTL_ORGNOFS?
       Purpose:    Queries the average, maximum, and minimum values among measured results for average count of origin offset.
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Origin_Offset_Total(ViSession instrSession, ViReal64 _VI_FAR *avg_dB, ViReal64 _VI_FAR *max_dB, ViReal64 _VI_FAR *min_dB)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "TTL_ORGNOFS?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%f,%f,%f", avg_dB, max_dB, min_dB);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Peak Code Domain Error Total
       Command(s): TTL_PCDERR?
       Purpose:    Queries the average, maximum, and minimum values among measured results for peak code domain error.
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Peak_Code_Domain_Error_Total(ViSession instrSession, ViReal64 _VI_FAR *avg_dB, ViReal64 _VI_FAR *max_dB, ViReal64 _VI_FAR *min_dB)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "TTL_PCDERR?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%f,%f,%f", avg_dB, max_dB, min_dB);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get RMS Phase Vector Error Total WCDMA
       Command(s): TTL_PHASEERR?
       Purpose:    Gets the total value among measured results for average count of phase vector error.
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_RMS_Phase_Vector_Error_Total_WCDMA(ViSession instrSession, ViReal64 _VI_FAR *avg_deg, ViReal64 _VI_FAR *max_deg, ViReal64 _VI_FAR *min_deg)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "TTL_PHASEERR?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%f,%f,%f", avg_deg, max_deg, min_deg);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get TX Power Total WCDMA
       Command(s): TTL_POWER?
       Purpose:    Queries the average, maximum, and minimum values among measured results for transmission power.
       Unit defined by:  AUMT8820_TX_POWER_UNIT_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_TX_Power_Total_WCDMA(ViSession instrSession, ViInt32 unit, ViReal64  _VI_FAR *average, ViReal64  _VI_FAR *maximum, ViReal64  _VI_FAR *minimum)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"DBM", "WATT", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   if (AuMT8820_InvalidViInt32Range(unit, 0, 1) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   len = sprintf(buf, "TTL_POWER? %s", commandArray[unit]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%f,%f,%f", average, maximum, minimum);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Peak Vector Error Total
       Command(s): TTL_PVECTERR?
       Purpose:    Queries the average, maximum, and minimum values among measured results for peak vector error..
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Peak_Vector_Error_Total(ViSession instrSession, ViReal64 _VI_FAR *avg_percent, ViReal64 _VI_FAR *max_percent, ViReal64 _VI_FAR *min_percent)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "TTL_PVECTERR?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%f,%f,%f", avg_percent, max_percent, min_percent);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Timing Error Total
       Command(s): TTL_TMGERR?
       Purpose:    Queries the average, maximum, and minimum values among measured results for timing error.
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Timing_Error_Total(ViSession instrSession, ViReal64 _VI_FAR *avg_chip, ViReal64 _VI_FAR *max_chip, ViReal64 _VI_FAR *min_chip)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "TTL_TMGERR?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%f,%f,%f", avg_chip, max_chip, min_chip);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Tx Measurement Status WCDMA
       Command(s): TXMSTAT?
       Purpose:    Get the current transmission measurement status
       Status defined by:  AUMT8820_MEASUREMENT_STATUS_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Tx_Measurement_Status_WCDMA(ViSession instrSession, ViInt32 _VI_FAR *status)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViChar    resBuf0[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "TXMSTAT?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   /* Get if numeric returned from instrument */
   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%d", status);

   /* Get if string returned from instrument */
   /*  Scan(rdBuf, "%s", resBuf0); */
   /*  if((auMT8820_status = AuMT8820_StringToEnum(commandArray, resBuf0, status)) < 0) */
   /*    return auMT8820_status;*/

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get TX Power Level
       Command(s): TXPWR?
       Purpose:    Queries the monitor value for transmission power.
       Unit defined by:  AUMT8820_TX_POWER_UNIT_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_TX_Power_Level_WCDMA(ViSession instrSession, ViInt32 unit, ViReal64 _VI_FAR *level)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"DBM", "WATT", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "TXPWR? %s", commandArray[unit]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%f", level);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Uplink DPCH Symbol Rate
       Command(s): UDPCHRATE?
       Purpose:    Queries the symbol rate for uplink DPCH.
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Uplink_DPCH_Symbol_Rate(ViSession instrSession, ViInt32 _VI_FAR *rate)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "UDPCHRATE?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%d", rate);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Uplink DPCH Scrambling Code
       Command(s): UDPCHSCRCODE[?]
       Purpose:    Sets the scrambling code for uplink DPCH.
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Uplink_DPCH_Scrambling_Code(ViSession instrSession, ViInt32 code)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   if (AuMT8820_InvalidViInt32Range(code, 0x00, 0x0FFFFFF) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   len = sprintf(buf, "UDPCHSCRCODE %X", code);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Uplink DPCH Scrambling Code
       Command(s): UDPCHSCRCODE[?]
       Purpose:    Queries the scrambling code for uplink DPCH.
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Uplink_DPCH_Scrambling_Code(ViSession instrSession, ViChar _VI_FAR *code)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "UDPCHSCRCODE?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%s", code);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Uplink DPCH Scrambling Code Type
       Command(s): UDPCHTYPE[?]
       Purpose:    Sets the scrambling code type for uplink DPCH.
       Type defined by:  AUMT8820_UPLINK_DPCH_SCRAMBLING_CODE_TYPE_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Uplink_DPCH_Scrambling_Code_Type(ViSession instrSession, ViInt32 type)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"LONG", "SHORT", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   if (AuMT8820_InvalidViInt32Range(type, 0, 1) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   len = sprintf(buf, "UDPCHTYPE %s", commandArray[type]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Uplink DPCH Scrambling Code Type
       Command(s): UDPCHTYPE[?]
       Purpose:    Sets the scrambling code type for uplink DPCH.
       Type defined by:  AUMT8820_UPLINK_DPCH_SCRAMBLING_CODE_TYPE_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Uplink_DPCH_Scrambling_Code_Type(ViSession instrSession, ViInt32  _VI_FAR *type)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"LONG", "SHORT", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViChar    resBuf0[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "UDPCHTYPE?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   /* Get if numeric returned from instrument */
   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   /* Scan(rdBuf, "%d", type); */

   /* Get if string returned from instrument */
   Scan(rdBuf, "%s", resBuf0); 
   if((auMT8820_status = AuMT8820_StringToEnum(commandArray, resBuf0, type)) < 0) 
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get IMSI
       Command(s): UE_IMSI?
       Purpose:    Queries the IMSI for a phone with its location registered.
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_IMSI(ViSession instrSession, ViChar _VI_FAR identity[])
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "UE_IMSI?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%s", identity);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get UE Power Class
       Command(s): UE_PWRCLASS?
       Purpose:    Queries the class of the power output from a phone.
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_UE_Power_Class(ViSession instrSession, ViInt32 _VI_FAR *class)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "UE_PWRCLASS?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%d", class);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Uplink External Loss
       Command(s): ULEXTLOSSW[?]
       Purpose:    Enables the external loss at the main I/O uplink.
       Mode defined by:  AUMT8820_UPLINK_EXTERNAL_LOSS_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Uplink_External_Loss(ViSession instrSession, ViInt32 mode)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"ON", "OFF", "COMMON", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   if (AuMT8820_InvalidViInt32Range(mode, 0, 2) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   len = sprintf(buf, "ULEXTLOSSW %s", commandArray[mode]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Uplink External Loss
       Command(s): ULEXTLOSSW[?]
       Purpose:    Enables the external loss at the main I/O uplink.
       Mode defined by:  AUMT8820_UPLINK_EXTERNAL_LOSS_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Uplink_External_Loss(ViSession instrSession, ViInt32  _VI_FAR *mode)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"ON", "OFF", "COMMON", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViChar    resBuf0[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "ULEXTLOSSW?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   /* Get if numeric returned from instrument */
   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   /* Scan(rdBuf, "%d", mode); */

   /* Get if string returned from instrument */
   Scan(rdBuf, "%s", resBuf0); 
   if((auMT8820_status = AuMT8820_StringToEnum(commandArray, resBuf0, mode)) < 0) 
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set External Up Link Loss WCDMA
       Command(s): ULEXTLOSS[?]
       Purpose:    Sets the external loss at the main I/O uplink
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_External_Up_Link_Loss_WCDMA(ViSession instrSession, ViReal64 level_dB)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   len = sprintf(buf, "ULEXTLOSS %.1f", level_dB);
   if (AuMT8820_InvalidViReal64Range(level_dB, -55.000000, 55.000000) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get External Up Link Loss WCDMA
       Command(s): ULEXTLOSS[?]
       Purpose:    Sets the external loss at the main I/O uplink
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_External_Up_Link_Loss_WCDMA(ViSession instrSession, ViReal64 _VI_FAR *level_dB)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "ULEXTLOSS?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%f", level_dB);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Uplink Frequency
       Command(s): ULFREQ[?],TFREQ,TXFREQ
       Purpose:    Get/Set the uplink frequency (resolution 1 Hz)
       Units defined by:  AUMT8820_FREQUENCY_UNIT_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Uplink_Frequency(ViSession instrSession, ViInt32 units, ViReal64 freq)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString unitString[] = {"HZ", "KHZ", "MHZ", "GHZ", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   if (AuMT8820_InvalidViReal64Range(freq, 0.0004000, 2700000000.0) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   if (AuMT8820_InvalidViInt32Range(units, 0, 3) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   len = sprintf(buf, "ULFREQ %.12f%s", freq, unitString[units]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Uplink Frequency
       Command(s): ULFREQ[?],TFREQ,TXFREQ
       Purpose:    Get/Set the uplink frequency (resolution 1 Hz)
       Units defined by:  AUMT8820_FREQUENCY_UNIT_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Uplink_Frequency(ViSession instrSession, ViReal64 _VI_FAR *freq)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViChar    resBuf0[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "ULFREQ?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%f%s", freq, resBuf0);

   /* Get units returned from instrument */
//    if((auMT8820_status = AuMT8820_StringToEnum(unitString, resBuf0)) < 0)
//      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Downlink Channel Level Unit
       Command(s): UNIT[?]
       Purpose:    Sets the setting unit for all channels to a relative value to the total power or to DPCH power.
       Object defined by:  AUMT8820_DOWNLINK_CHANNEL_LEVEL_UNIT_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Downlink_Channel_Level_Unit(ViSession instrSession, ViInt32 object)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"IOR", "DPCH", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   if (AuMT8820_InvalidViInt32Range(object, 0, 1) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   len = sprintf(buf, "UNIT %s", commandArray[object]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Downlink Channel Level Unit
       Command(s): UNIT[?]
       Purpose:    Sets the setting unit for all channels to a relative value to the total power or to DPCH power.
       Object defined by:  AUMT8820_DOWNLINK_CHANNEL_LEVEL_UNIT_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Downlink_Channel_Level_Unit(ViSession instrSession, ViInt32  _VI_FAR *object)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"IOR", "DPCH", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViChar    resBuf0[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "UNIT?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   /* Get if numeric returned from instrument */
   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   /* Scan(rdBuf, "%d", object); */

   /* Get if string returned from instrument */
   Scan(rdBuf, "%s", resBuf0); 
   if((auMT8820_status = AuMT8820_StringToEnum(commandArray, resBuf0, object)) < 0) 
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set URA Identity
       Command(s): URAID[?]
       Purpose:    Sets the URA identification code for a terminal
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_URA_Identity(ViSession instrSession, ViInt32 iD)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   if (AuMT8820_InvalidViInt32Range(iD,0x0, 0x0FFFF) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   len = sprintf(buf, "URAID %04X", iD);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get URA Identity
       Command(s): URAID[?]
       Purpose:    Sets the URA identification code for a terminal
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_URA_Identity(ViSession instrSession,  ViChar _VI_FAR *iD)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "URAID?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%s", iD);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Voltage Display  WCDMA
       Command(s): VDSPL[?]
       Purpose:    Sets/Gets the power voltage display units when the RF output setting is in db.
       Unit defined by:  AUMT8820_VOLTAGE_DISPLAY_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Voltage_Display_WCDMA(ViSession instrSession, ViInt32 unit)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"EMF", "TERM", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   if (AuMT8820_InvalidViInt32Range(unit, 0, 1) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   len = sprintf(buf, "VDSPL %s", commandArray[unit]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Voltage Display WCDMA
       Command(s): VDSPL[?]
       Purpose:    Sets/Gets the power voltage display units when the RF output setting is in db.
       Unit defined by:  AUMT8820_VOLTAGE_DISPLAY_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Voltage_Display_WCDMA(ViSession instrSession, ViInt32  _VI_FAR *unit)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"EMF", "TERM", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViChar    resBuf0[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "VDSPL?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   /* Get if numeric returned from instrument */
   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   /* Scan(rdBuf, "%d", unit); */

   /* Get if string returned from instrument */
   Scan(rdBuf, "%s", resBuf0); 
   if((auMT8820_status = AuMT8820_StringToEnum(commandArray, resBuf0, unit)) < 0) 
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Video Filter Length
       Command(s): VFILTLEN[?]
       Purpose:    Sets/Gets the video filter length.
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Video_Filter_Length(ViSession instrSession, ViReal64 length_us)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   len = sprintf(buf, "VFILTLEN %.1fUS", length_us);
   if (AuMT8820_InvalidViReal64Range(length_us, 0.100000, 1066.700000) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Video Filter Length
       Command(s): VFILTLEN[?]
       Purpose:    Sets/Gets the video filter length.
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Video_Filter_Length(ViSession instrSession, ViReal64 _VI_FAR *length_us)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "VFILTLEN?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%f", length_us);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Voice Channel
       Command(s): VOICECH[?]
       Purpose:    Sets the voice channel to be measured for bit error rate.
       Subflow defined by:  AUMT8820_VOICE_CHANNEL_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Voice_Channel(ViSession instrSession, ViInt32 subflow)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"SUBFLOW1", "SUBFLOW2", "SUBFLOW3", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   if (AuMT8820_InvalidViInt32Range(subflow, 0, 2) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   len = sprintf(buf, "VOICECH %s", commandArray[subflow]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Voice Channel
       Command(s): VOICECH[?]
       Purpose:    Sets the voice channel to be measured for bit error rate.
       Subflow defined by:  AUMT8820_VOICE_CHANNEL_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Voice_Channel(ViSession instrSession, ViInt32  _VI_FAR *subflow)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"SUBFLOW1", "SUBFLOW2", "SUBFLOW3", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViChar    resBuf0[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "VOICECH?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   /* Get if numeric returned from instrument */
   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   /* Scan(rdBuf, "%d", subflow); */

   /* Get if string returned from instrument */
   Scan(rdBuf, "%s", resBuf0); 
   if((auMT8820_status = AuMT8820_StringToEnum(commandArray, resBuf0, subflow)) < 0) 
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Modulation Analysis Vertical Scale
       Command(s): VSCALE_MOD[?]
       Purpose:    Sets the full scale for the waveform displayed in the modulation analysis view window
       Range defined by:  AUMT8820_MODULATION_ANALYSIS_VERTICAL_SCALE_RANGE_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Modulation_Analysis_Vertical_Scale(ViSession instrSession, ViInt32 range)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"5", "10", "20", "50", "100", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   if (AuMT8820_InvalidViInt32Range(range, 0, 4) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   len = sprintf(buf, "VSCALE_MOD %s", commandArray[range]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Modulation Analysis Vertical Scale
       Command(s): VSCALE_MOD[?]
       Purpose:    Sets the full scale for the waveform displayed in the modulation analysis view window
       Range defined by:  AUMT8820_MODULATION_ANALYSIS_VERTICAL_SCALE_RANGE_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Modulation_Analysis_Vertical_Scale(ViSession instrSession, ViInt32  _VI_FAR *range)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"5", "10", "20", "50", "100", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViChar    resBuf0[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "VSCALE_MOD?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   /* Get if numeric returned from instrument */
   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   /* Scan(rdBuf, "%d", range); */

   /* Get if string returned from instrument */
   Scan(rdBuf, "%s", resBuf0); 
   if((auMT8820_status = AuMT8820_StringToEnum(commandArray, resBuf0, range)) < 0) 
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Video Trigger Treshold
       Command(s): VTRGTHRESH[?]
       Purpose:    Sets the video trigger threshold for time domain measurements

       NOT IMPLEMENTED ON MT8820
*/
/*****************************************************************************/
//ViStatus _VI_FUNC AuMT8820_Set_Video_Trigger_Treshold(ViSession instrSession, ViInt32 level)
//{
//   ViStatus  auMT8820_status = VI_SUCCESS;
//   ViInt32   len;
//   ViUInt32  retCnt = 0;
//   ViChar    buf[BUFFER_SIZE + 1];
//   
//   if (AuMT8820_InvalidViInt32Range(level,-40,0) == VI_TRUE)
//      return VI_ERROR_OUTOF_RANGE;
//   len = sprintf(buf, "VTRGTHRESH %d", level);
//   
//   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
//      return auMT8820_status;
//
//   return auMT8820_status;
//
//}
//

/*****************************************************************************/
/*
       Function:   Get Video Trigger Treshold
       Command(s): VTRGTHRESH[?]
       Purpose:    Sets the video trigger threshold for time domain measurements

       NOT IMPLEMENTED ON MT8820
*/
/*****************************************************************************/
//ViStatus _VI_FUNC AuMT8820_Get_Video_Trigger_Treshold(ViSession instrSession,  ViInt32 _VI_FAR *level)
//{
//   ViStatus  auMT8820_status = VI_SUCCESS;
//   ViInt32   len;
//   ViUInt32  retCnt = 0;
//   ViChar    buf[BUFFER_SIZE + 1];
//   ViChar    rdBuf[BUFFER_SIZE];
//   ViUInt32  bytesRead;
//   
//   len = sprintf(buf, "VTRGTHRESH?");
//   
//   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
//      return auMT8820_status;
//
//   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
//      return auMT8820_status;
//   Scan(rdBuf, "%d", level);
//
//   return auMT8820_status;
//
//}
//

/*****************************************************************************/
/*
       Function:   View Window Close
       Command(s): VWCLS
       Purpose:    Closes the view window
       View defined by:  AUMT8820_VIEW_WINDOW_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_View_Window_Close(ViSession instrSession, ViInt32 view)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"OBW", "SMASK", "MOD", "NONE",  0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   if (AuMT8820_InvalidViInt32Range(view, 0, 3) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   len = sprintf(buf, "VWCLS %s", commandArray[view]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set View Window Open
       Command(s): VWOPEN[?],VWOPN[?]
       Purpose:    Opens the view window.
       View defined by:  AUMT8820_VIEW_WINDOW_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_View_Window_Open(ViSession instrSession, ViInt32 view)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"OBW", "SMASK", "MOD", "NONE",  0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   if (AuMT8820_InvalidViInt32Range(view, 0, 3) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   len = sprintf(buf, "VWOPN %s", commandArray[view]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get View Window Open
       Command(s): VWOPEN[?],VWOPN[?]
       Purpose:    Opens the view window.
       View defined by:  AUMT8820_VIEW_WINDOW_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_View_Window_Open(ViSession instrSession, ViInt32  _VI_FAR *view)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"OBW", "SMASK", "MOD", "NONE", "",  0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViChar    resBuf0[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "VWOPN?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   /* Get if numeric returned from instrument */
   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   /* Scan(rdBuf, "%d", view); */

   /* Get if string returned from instrument */
   Scan(rdBuf, "%s", resBuf0); 
   if((auMT8820_status = AuMT8820_StringToEnum(commandArray, resBuf0, view)) < 0) 
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Window Size WCDMA
       Command(s): WINSIZE[?]
       Purpose:    Changes the size of the active window.
       Size defined by:  AUMT8820_WINDOW_SIZE_xxx
       Defined in MT8820 GSM LabWindows Driver.c
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Window_Size_WCDMA(ViSession instrSession, ViInt32 size)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"100", "70", "50", "30", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   if (AuMT8820_InvalidViInt32Range(size, 0, 3) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   len = sprintf(buf, "WINSIZE %s", commandArray[size]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Window Size WCDMA
       Command(s): WINSIZE[?]
       Purpose:    Changes the size of the active window.
       Size defined by:  AUMT8820_WINDOW_SIZE_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Window_Size_WCDMA(ViSession instrSession, ViInt32  _VI_FAR *size)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"100", "70", "50", "30", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViChar    resBuf0[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "WINSIZE?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   /* Get if numeric returned from instrument */
   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   /* Scan(rdBuf, "%d", size); */

   /* Get if string returned from instrument */
   Scan(rdBuf, "%s", resBuf0); 
   if((auMT8820_status = AuMT8820_StringToEnum(commandArray, resBuf0, size)) < 0) 
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Occupied Bandwidth Spectrum Data
       Command(s): XMO[?]
       Purpose:    Sets the spectrum data for occupied bandwidth measurement and or stores it to the MT8820A
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Occupied_Bandwidth_Spectrum_Data(ViSession instrSession, ViInt32 position, ViInt32 _VI_FAR data)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   if (AuMT8820_InvalidViInt32Range(position,0,280) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   len = sprintf(buf, "XMO %d,%d", position, data);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Occupied Bandwidth Spectrum Data
       Command(s): XMO[?]
       Purpose:    Sets the spectrum data for occupied bandwidth measurement and or stores it to the MT8820A
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Occupied_Bandwidth_Spectrum_Data(ViSession instrSession, ViInt32 position, ViInt32 length,  ViChar  *data)
{
 ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   if (AuMT8820_InvalidViInt32Range(position,0,280) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   if (AuMT8820_InvalidViInt32Range(length,1,281) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   len = sprintf(buf, "XMO? %d,%d", position, length);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%s", data);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Spectrum Emission Spectrum Data
       Command(s): XMS[?]
       Purpose:    Sets the spectrum data for spectrum emission measurement and or stores it to the MT8820A
*
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Spectrum_Emission_Spectrum_Data(ViSession instrSession, ViInt32 position, ViInt32 _VI_FAR data)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   if (AuMT8820_InvalidViInt32Range(position,0,278) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   len = sprintf(buf, "XMS %d,%d", position, data);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Spectrum Emission Spectrum Data
       Command(s): XMS[?]
       Purpose:    Sets the spectrum data for spectrum emission measurement and or stores it to the MT8820A
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Spectrum_Emission_Spectrum_Data(ViSession instrSession, ViInt32 position, ViInt32 length,  ViChar _VI_FAR *data)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   if (AuMT8820_InvalidViInt32Range(position,0,278) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   if (AuMT8820_InvalidViInt32Range(length,1,279) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   len = sprintf(buf, "XMS? %d,%d", position, length);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%s", data);

   return auMT8820_status;

}

/*****************************************************************************/
/*
       Function:   Set Spectrum Monitor Zone Marker Frequency
       Command(s): ZMKRF_SPM
       Purpose:    Queries the marker point frequency of the zone marker in the spectrum monitor screen.
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Spectrum_Monitor_Zone_Marker_Frequency(ViSession instrSession, ViReal64 _VI_FAR frequency_Hz)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "ZMKRP_SPM %fMHZ",frequency_Hz);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
   return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Spectrum Monitor Zone Marker Frequency
       Command(s): ZMKRF_SPM?
       Purpose:    Queries the marker point frequency of the zone marker in the spectrum monitor screen.
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Spectrum_Monitor_Zone_Marker_Frequency(ViSession instrSession, ViReal64 _VI_FAR *frequency_Hz)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "ZMKRF_SPM?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%f", frequency_Hz);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Spectrum Monitor Zone Marker Level
       Command(s): ZMKRL_SPM?
       Purpose:    Queries the marker point level of the zone marker in the spectrum monitor screen.
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Spectrum_Monitor_Zone_Marker_Level(ViSession instrSession, ViReal64 _VI_FAR *level_dB)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "ZMKRL_SPM?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%f", level_dB);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Spectrum Monitor Zone Marker Position
       Command(s): ZMKRP_SPM[?]
       Purpose:    Queries the marker point frequency of the zone marker in the spectrum monitor screen.
*/
/*****************************************************************************/
//ViStatus _VI_FUNC AuMT8820_Set_Spectrum_Monitor_Zone_Marker_Position(ViSession instrSession, ViReal64 frequency_MHz)
//{
//   ViStatus  auMT8820_status = VI_SUCCESS;
//   ViInt32   len;
//   ViUInt32  retCnt = 0;
//   ViChar    buf[BUFFER_SIZE + 1];
//   
//   len = sprintf(buf, "ZMKRP_SPM %.4fMHZ", frequency_MHz);
//   if (AuMT8820_InvalidViReal64Range(frequency, -10.850000, 2711.250000) == VI_TRUE)
//      return VI_ERROR_OUTOF_RANGE;
//   
//   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
//      return auMT8820_status;
//
//   return auMT8820_status;
//
//}


/*****************************************************************************/
/*
       Function:   Get Spectrum Monitor Zone Marker Position
       Command(s): ZMKRP_SPM[?]
       Purpose:    Queries the marker point frequency of the zone marker in the spectrum monitor screen.
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Spectrum_Monitor_Zone_Marker_Position(ViSession instrSession, ViReal64 _VI_FAR *frequency_MHz)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "ZMKRP_SPM?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   Scan(rdBuf, "%f", frequency_MHz);

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Set Spectrum Monitor Zone Marker
       Command(s): ZMKR_SPM[?]
       Purpose:    Sets the zone marker display in the spectrum monitor screen.
       Setting defined by:  AUMT8820_SPECTRUM_MONITOR_ZONE_MARKER_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Set_Spectrum_Monitor_Zone_Marker(ViSession instrSession, ViInt32 setting)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"ON", "OFF", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   
   if (AuMT8820_InvalidViInt32Range(setting, 0, 1) == VI_TRUE)
      return VI_ERROR_OUTOF_RANGE;
   len = sprintf(buf, "ZMKR_SPM %s", commandArray[setting]);
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*
       Function:   Get Spectrum Monitor Zone Marker
       Command(s): ZMKR_SPM[?]
       Purpose:    Sets the zone marker display in the spectrum monitor screen.
       Setting defined by:  AUMT8820_SPECTRUM_MONITOR_ZONE_MARKER_xxx
*/
/*****************************************************************************/
ViStatus _VI_FUNC AuMT8820_Get_Spectrum_Monitor_Zone_Marker(ViSession instrSession, ViInt32  _VI_FAR *setting)
{
   ViStatus  auMT8820_status = VI_SUCCESS;
   static ViString commandArray[] = {"ON", "OFF", 0L};
   ViInt32   len;
   ViUInt32  retCnt = 0;
   ViChar    buf[BUFFER_SIZE + 1];
   ViChar    rdBuf[BUFFER_SIZE];
   ViChar    resBuf0[BUFFER_SIZE];
   ViUInt32  bytesRead;
   
   len = sprintf(buf, "ZMKR_SPM?");
   
   if((auMT8820_status = AuMT8820_Write(instrSession, buf, len, &retCnt)) < 0)
      return auMT8820_status;

   /* Get if numeric returned from instrument */
   if((auMT8820_status = AuMT8820_Read(instrSession, rdBuf, SMALL_BUFFER, &bytesRead)) < 0)
      return auMT8820_status;
   /* Scan(rdBuf, "%d", setting); */

   /* Get if string returned from instrument */
   Scan(rdBuf, "%s", resBuf0); 
   if((auMT8820_status = AuMT8820_StringToEnum(commandArray, resBuf0, setting)) < 0) 
      return auMT8820_status;

   return auMT8820_status;

}


/*****************************************************************************/
/*=== END INSTRUMENT DRIVER SOURCE CODE =====================================*/
/*****************************************************************************/
